import store from 'mod_otopo/items/store';
import ajax from 'mod_otopo/items/ajax';
import Degree from 'mod_otopo/items/components/degree';
import ColorPicker from 'mod_otopo/color';
import utils from 'mod_otopo/utils';

var Item = {
    components: {
        'Degree': Degree,
        'ColorPicker': ColorPicker
    },
    props: {
        otopo: {
            type: Number
        },
        item: {
            type: Object
        },
        index: {
            type: Number
        }
    },
    mounted() {
        this.$refs.name.focus();
    },
    data: function () {
        return {
            nbr: this.item.degrees.length,
            strings: this.$root.$data.strings,
            hasOtopo: this.$root.$data.hasOtopo,
            cmid: this.$root.$data.cmid,
            processChange: utils.debounce(() => this.saveItem()),
            validated: false,
            creationPending: false,
            collapsed: false,
            state: this.$root.$data.state
        };
    },
    methods: {
        deleteItem() {
            if(!this.hasOtopo) {
                store.deleteItem(this.index);
                ajax.deleteItem(this.item.id, this.cmid);
            }
        },
        duplicateItem(e) {
            e.preventDefault();
            if(!this.hasOtopo) {
                const item = {...this.item};
                item.degrees = [];
                this.item.degrees.forEach((degree) => {
                    item.degrees.push({...degree});
                });
                delete item.id;
                item.degrees.forEach((degree) => {
                    delete degree.id;
                });
                item.ord = this.item.ord + 1;
                var itemsToPersist = store.addItemAfter(this.index, item);
                itemsToPersist.forEach((item) => {
                    if(item.id) {
                        var itemToPersist = {...item};
                        delete itemToPersist.degrees;
                        ajax.editItem(itemToPersist, this.cmid);
                    }
                });
                const itemToCreate = {...item};
                delete itemToCreate.degrees;
                ajax.createItem(this.otopo, itemToCreate, this.cmid).then((itemId) => {
                    item.id = itemId;
                    item.degrees.forEach((degree) => {
                        ajax.createDegree(itemId, degree, this.cmid).then((degreeId) => {
                            degree.id = degreeId;
                        });
                    });
                });
            }
        },
        addDegree(e) {
            e.preventDefault();
            if(!this.hasOtopo) {
                store.addDegreeToItem(
                    this.item,
                    {
                        'id': null,
                        'name': '',
                        'description': '',
                        'grade': this.nbr + 1,
                        'ord': this.item.degrees.length > 0 ? this.item.degrees[this.item.degrees.length - 1].ord + 1 : 0
                    }
                );
                ++this.nbr;
            }
        },
        saveItem() {
            this.validated = true;
            if(this.$refs.formItem.checkValidity()) {
                const item = {...this.item};
                delete item.degrees;
                if(item.id) {
                    ajax.editItem(item, this.cmid);
                }
                else {
                    if(!this.creationPending) {
                        this.creationPending = true;
                        delete item.id;
                        ajax.createItem(this.otopo, item, this.cmid).then((itemId) => {
                            this.item.id = itemId;
                            this.creationPending = false;
                        });
                    }
                }
            }
        },
        startDrag (evt, index) {
            if(!this.hasOtopo) {
                evt.dataTransfer.dropEffect = 'move';
                evt.dataTransfer.effectAllowed = 'move';
                evt.dataTransfer.setData('itemIndex', index);
                store.startDraggingItem();
            }
        },
        endDrag () {
            store.stopDraggingItem();
        },
        onDrop (evt, before) {
            evt.preventDefault();
            if(!this.hasOtopo) {
                const itemIndex = parseInt(evt.dataTransfer.getData('itemIndex'));
                const testItemIndex = before ? itemIndex + 1 : itemIndex;
                const testIndex = before ? this.index : this.index + 1;
                if(itemIndex != this.index && testItemIndex != testIndex) {
                    var deletedItem = store.deleteItem(itemIndex);
                    deletedItem.ord = before ? this.item.ord : this.item.ord + 1;
                    var itemsToPersist = store.addItemAfter(this.index, deletedItem);
                    itemsToPersist.forEach((item) => {
                        if(item.id) {
                            var itemToPersist = {...item};
                            delete itemToPersist.degrees;
                            ajax.editItem(itemToPersist, this.cmid);
                        }
                    });
                    if(deletedItem.id) {
                        var deletedItemToPersist = {...deletedItem};
                        delete deletedItemToPersist.degrees;
                        ajax.editItem(deletedItemToPersist, this.cmid);
                    }
                }
            }
        },
        onDropDegree (evt) {
            evt.preventDefault();
            if(!this.hasOtopo) {
                const degreeIndex = parseInt(evt.dataTransfer.getData('degreeIndex'));
                const itemIndex = parseInt(evt.dataTransfer.getData('itemIndex'));

                var deletedDegree = store.deleteDegreeFromItem(this.$root.$data.state.items[itemIndex], degreeIndex);
                deletedDegree.ord = 0;
                store.addDegreeToItem(this.item, deletedDegree);

                if(deletedDegree.id) {
                    ajax.deleteDegree(deletedDegree.id, this.cmid);
                    var deletedDegreeToCreate = {...deletedDegree};
                    delete deletedDegreeToCreate.id;
                    ajax.createDegree(this.item.id, deletedDegreeToCreate, this.cmid).then((degreeId) => {
                        store.degreeCreated(deletedDegree, degreeId);
                    });
                }
            }
        },
        collapse (e) {
            e.preventDefault();
            this.collapsed = !this.collapsed;
        }
    },
    computed: {
        isDragging() {
            return this.state.draggingItem;
        },
        isDraggingDegree() {
            return this.state.draggingDegree;
        }
    },
    template: `
<div class="item">
    <div
        class="drop-zone"
        v-bind:class="{'bg-otopo-light': isDragging}"
        @dragover.prevent
        @dragenter.prevent
        @drop="onDrop($event, true)">
    </div>
    <div
        class="row border rounded bg-light pt-2"
    >
        <div
            class="col-md-12"
            draggable
            @dragstart="startDrag($event, index)"
            @dragend="endDrag"
        >
            <a class="dropdown-toggle nav-link drag-title" href="#" v-on:click="collapse">
                <i class="icon fa fa-arrows"></i>
                <label
                    class="font-weight-bold text-dark"
                    :for="'item_name_' + index"
                >
                    {{strings.item}} {{index+1}}
                </label>
            </a>
        </div>
        <div class="col-md-5" v-if="!collapsed">
            <form ref="formItem" v-bind:class="{ 'was-validated': validated }">
            <div class="input-group mb-3">
                <input
                    v-model="item.name"
                    ref="name"
                    class="form-control"
                    type="text"
                    maxlength="255"
                    :title="strings.string_limit_255"
                    :id="'item_name_' + index"
                    :disabled="creationPending"
                    @input="processChange()"
                    required
                >
            </div>
            <label
                class="font-weight-bold text-dark"
                :for="'item_color_' + index"
            >
                {{strings.choose_item_color}}
            </label>
            <div class="input-group mb-3" :id="'item_color_colorpicker_' + index">
                <ColorPicker
                    :color="item.color"
                    v-model="item.color"
                    :id="'item_color_' + index"
                    @input="processChange()"
                    :disabled="creationPending"
                    required
                />
            </div>
            <button
                v-on:click="duplicateItem"
                :disabled="creationPending || hasOtopo"
                class="btn btn-primary mt-2 mb-3"
            >
                {{ strings.duplicate_item }}
            </button>
            <button
                v-on:click="deleteItem"
                :disabled="creationPending || hasOtopo"
                class="btn btn-danger mt-2 mb-3"
            >
                {{ strings.delete_item }}
            </button>
            </form>
        </div>
        <div class="col-md-7" v-if="!collapsed">
            <Degree
               v-for="(degree, degreeIndex) in item.degrees"
               :key="degree.id ? degree.id : 'new_degree' + degreeIndex"
               :item="item"
               :degree="degree"
               :index="degreeIndex"
               :itemIndex="index"
            />
            <a
                v-on:click="addDegree"
                :disabled="creationPending"
                class="mt-2 mb-3 d-inline-block"
                v-bind:class="{'disabled': hasOtopo}"
                href="#"
            >
                + {{ strings.add_degree }}
            </a>
            <div
                v-if="item.degrees.length === 0"
                class="drop-zone"
                v-bind:class="{'bg-otopo-light': isDraggingDegree}"
                @dropover.prevent
                @dragenter.prevent
                @drop="onDropDegree($event)">
            </div>
        </div>
    </div>
    <div
        class="drop-zone"
        v-bind:class="{'bg-otopo-light': isDragging}"
        @dragover.prevent
        @dragenter.prevent
        @drop="onDrop($event, false)">
    </div>
</div>
`
};

export default Item;
