<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Otopo functions
 *
 * @package     mod_otopo
 * @copyright   2022 Kosmos <moodle@kosmos.fr>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

function toform_from_sessions($sessions) {
    if(count($sessions) == 0) {
        return null;
    }
    $toform = new stdClass();
    $i = 0;
    foreach($sessions as $session) {
        $toform->id[$i] = $session->id;
        $toform->name[$i] = $session->name;
        $toform->color[$i] = $session->color;
        $toform->allowsubmissionfromdate[$i] = $session->allowsubmissionfromdate;
        $toform->allowsubmissiontodate[$i] = $session->allowsubmissiontodate;
        $i++;
    }
    return $toform;
}

function get_items_from_otopo($otopo_id) {
    global $DB;

    $items = [];
    $rs = $DB->get_recordset_sql('SELECT item.id, item.name, item.color, item.ord,  degree.id AS did, degree.name AS dname, degree.description AS ddescription, degree.grade AS dgrade, degree.ord AS dord FROM {otopo_item} item LEFT JOIN {otopo_item_degree} degree ON degree.item = item.id WHERE item.otopo = :otopo ORDER BY item.id, degree.ord', ['otopo' => $otopo_id]);
    foreach ($rs as $record) {
        if(!array_key_exists($record->id, $items)) {
            $items[$record->id] = (object)['id' => $record->id, 'name' => $record->name, 'color' => empty($record->color) ? '#000000' : $record->color, 'ord' => $record->ord, 'degrees' => []];
        }
        if($record->did) {
            $items[$record->id]->degrees[] = (object)['id' => $record->did, 'name' => $record->dname, 'description' => $record->ddescription, 'grade' => $record->dgrade, 'ord' => $record->dord];
        }
    }
    $rs->close();

    return $items;
}

function get_items_sorted_from_otopo($otopo_id) {
    global $DB;

    $items = [];
    $rs = $DB->get_recordset_sql('SELECT item.id, item.name, item.color, item.ord,  degree.id AS did, degree.name AS dname, degree.description AS ddescription, degree.grade AS dgrade, degree.ord AS dord FROM {otopo_item} item LEFT JOIN {otopo_item_degree} degree ON degree.item = item.id WHERE item.otopo = :otopo ORDER BY item.ord, degree.ord', ['otopo' => $otopo_id]);
    foreach ($rs as $record) {
        if(!array_key_exists($record->ord, $items)) {
            $items[$record->ord] = (object)['id' => $record->id, 'name' => $record->name, 'color' => empty($record->color) ? '#000000' : $record->color, 'ord' => $record->ord, 'degrees' => []];
        }
        if($record->did) {
            $items[$record->ord]->degrees[] = (object)['id' => $record->did, 'name' => $record->dname, 'description' => $record->ddescription, 'grade' => $record->dgrade, 'ord' => $record->dord];
        }
    }
    $rs->close();

    return array_values($items);
}

function table_items(&$items) {
    $nbr_degrees_max = 0;
    foreach($items as $item) {
        if(count($item->degrees) > $nbr_degrees_max) {
            $nbr_degrees_max = count($item->degrees);
        }
        $item->one_degree_has_desc = false;
        foreach($item->degrees as $index => $degree) {
            $degree->index = $index + 1;
            if(!empty($degree->description)) {
                $item->one_degree_has_desc = true;
            }
            $degree->description = nl2br($degree->description);
        }
    }

    return $nbr_degrees_max;
}

function get_templates() {
    global $DB;
    return $DB->get_records('otopo_template', null, 'name');
}


function copy_items($otopo_id, $items) {
    global $DB;

    foreach($items as $item) {
        $degrees = $item->degrees;
        unset($item->id);
        unset($item->degrees);
        $item->otopo = $otopo_id;
        $item->id = $DB->insert_record('otopo_item', $item);
        foreach($degrees as $degree) {
            unset($degree->id);
            $degree->item = $item->id;
            $DB->insert_record('otopo_item_degree', $degree);
        }
    }
}

function delete_items($otopo_id) {
    global $DB;

    $items = $DB->get_records('otopo_item', array('otopo' => $otopo_id), 'id');
    foreach($items as $item) {
        $DB->delete_records('otopo_item_degree', array('item' => $item->id));
    }
    $DB->delete_records('otopo_item', array('otopo' => $otopo_id));
}

function get_user_otopos($otopo, $user) {
    global $DB;

    $otopos = [];

    $items = get_items_from_otopo($otopo->id);

    if(count($items) > 0) {
        list($insql, $params) = $DB->get_in_or_equal(array_keys($items));
        $params[] = intval($user->id);

        if($otopo->session) {
            $sql = "SELECT * FROM {otopo_user_otopo} WHERE item $insql AND userid = ? AND session > 0";
        }
        else {
            $sql = "SELECT * FROM {otopo_user_otopo} WHERE item $insql AND userid = ? AND session < 0";
        }
        $my_otopos = $DB->get_records_sql($sql, $params);

        foreach($my_otopos as $my_otopo) {
            if(!array_key_exists($my_otopo->item, $otopos)) {
                $otopos[$my_otopo->item] = array();
            }
            $otopos[$my_otopo->item][$my_otopo->session] = $my_otopo;
        }
    }

    return $otopos;
}

function get_users_otopos($otopo, $users_ids) {
    global $DB;

    $otopos = [];

    $items = get_items_from_otopo($otopo->id);

    if(count($items) > 0 && count($users_ids) > 0) {
        list($insql, $params) = $DB->get_in_or_equal(array_keys($items));
        list($insql2, $params2) = $DB->get_in_or_equal($users_ids);
        $params = array_merge($params, $params2);

        if($otopo->session) {
            $sql = "SELECT * FROM {otopo_user_otopo} WHERE item $insql AND userid $insql2 AND session > 0";
        }
        else {
            $sql = "SELECT * FROM {otopo_user_otopo} WHERE item $insql AND userid $insql2 AND session < 0";
        }
        $users_otopos = $DB->get_records_sql($sql, $params);

        foreach($users_otopos as $users_otopo) {
            if (!array_key_exists($users_otopo->userid, $otopos)) {
                $otopos[$users_otopo->userid] = array();
            }
            if (!array_key_exists($users_otopo->session, $otopos[$users_otopo->userid])) {
                $otopos[$users_otopo->userid][$users_otopo->session] = array();
            }
            $otopos[$users_otopo->userid][$users_otopo->session][$users_otopo->item] = $users_otopo;
        }
    }

    return $otopos;
}

function get_users_sessions_with_otopos($otopo) {
    global $DB;

    $otopos = [];

    $items = get_items_from_otopo($otopo->id);

    if(count($items) > 0) {
        list($insql, $params) = $DB->get_in_or_equal(array_keys($items));

        if($otopo->session) {
            $sql = "SELECT * FROM {otopo_user_otopo} WHERE item $insql AND session > 0";
        }
        else {
            $sql = "SELECT * FROM {otopo_user_otopo} WHERE item $insql AND session < 0";
        }
        $users_otopos = $DB->get_records_sql($sql, $params);

        foreach($users_otopos as $user_otopo) {
            if (!array_key_exists($user_otopo->userid, $otopos)) {
                $otopos[$user_otopo->userid] = array();
            }
            if (!in_array($user_otopo->session, $otopos[$user_otopo->userid])) {
                $otopos[$user_otopo->userid][$user_otopo->session] = ['id' => $user_otopo->session, 'validated' => session_is_valid_or_closed($otopo->id, (object)array('id' => $user_otopo->userid), $user_otopo->session)];
            }
            $user_otopo->item = $items[$user_otopo->item];
        }
    }

    return $otopos;
}

function get_graders($otopo) {
    global $DB;

    $graders = $DB->get_records('otopo_grader', array('otopo' => $otopo->id));

    $result = [];
    foreach($graders as $grader) {
        if (!array_key_exists($grader->userid, $result)) {
            $result[$grader->userid] = [];
        }
        $result[$grader->userid][$grader->session] = $grader;
    }

    return $result;
}

function convert_grade_to_gradebook($user, $grade, $comment) {
    return array(
        'userid' => $user,
        'rawgrade' => $grade,
        'feedback' => strip_tags($comment),
    );
}

function get_participants($o, $context) {
    global $DB;

    list($esql, $params) = get_enrolled_sql($context, 'mod/otopo:fill');

    $fields = 'u.*';
    $orderby = 'u.lastname, u.firstname, u.id';
    $sql = "SELECT $fields
                      FROM {user} u
                      JOIN ($esql) je ON je.id = u.id
                     WHERE u.deleted = 0
                  ORDER BY $orderby";

    $participants = $DB->get_records_sql($sql, $params);

    return $participants;
}

function get_participant($o, $context, $userid) {
    global $DB;

    list($esql, $params) = get_enrolled_sql($context, 'mod/otopo:fill');
    $params['userid'] = $userid;

    $fields = 'u.*';
    $orderby = 'u.lastname, u.firstname, u.id';
    $sql = "SELECT $fields
                      FROM {user} u
                      JOIN ($esql) je ON je.id = u.id
                     WHERE u.deleted = 0 AND u.id = :userid
                  ORDER BY $orderby";

    $participant = $DB->get_record_sql($sql, $params);

    $otopos = get_users_sessions_with_otopos($o);

    $participant->fullname = fullname($participant, has_capability('moodle/site:viewfullnames', $context));
    if(array_key_exists($participant->id, $otopos)) {
        $key = 1;
        foreach($otopos[$participant->id] as $otopo) {
            $otopo['key'] = $key;
            $participant->sessions[] = $otopo;
            ++$key;
        }
    }

    return $participant;
}

function get_distribution_by_item($o, $users, $session = null) {
    $otopos = get_users_otopos($o, $users);

    $last_otopos = [];

    if($o->session) {
        if(!$session) {
            $session = get_last_session_closed($o)->id;
        }
        if($session) {
            foreach($otopos as $user => $otopo) {
                if(array_key_exists($session, $otopo)) {
                    $last_otopos[$user] = $otopo[$session];
                }
            }
        }
        else {
            return [];
        }
    }
    else {
        foreach($otopos as $user => $session_otopo) {
            foreach(array_reverse($session_otopo, true) as $session => $otopo) {
                if(session_is_valid($o->id, (object)['id' => $user], $session)) {
                    $last_otopos[$user] = $otopo;
                    break;
                }
            }
        }
    }

    $distribution = [];

    if(!empty($last_otopos)) {
        $items = get_items_sorted_from_otopo($o->id);

        foreach($items as $key1 => $item) {
            foreach($item->degrees as $key2 => $degree) {
                if(!array_key_exists($key2, $distribution)) {
                    $distribution[$key2] = [];
                }
                if(!array_key_exists($key1, $distribution[$key2])) {
                    $distribution[$key2][$key1] = 0;
                }
                foreach($last_otopos as $user => $otopo) {
                    if(array_key_exists($item->id, $otopo) && $otopo[$item->id]->degree == $degree->id) {
                        $distribution[$key2][$key1] += 1;
                    }
                }
            }
        }
    }

    return $distribution;
}

function has_otopo($otopo_id) {
    global $DB;
    return $DB->record_exists_sql('SELECT * FROM {otopo_user_otopo} INNER JOIN {otopo_item} AS it ON item = it.id WHERE it.otopo = :otopo', array('otopo' => $otopo_id));
}

function get_sessions($otopo) {
    global $DB;

    $sessions = array();

    $sessions_list = $DB->get_records('otopo_session', array('otopo' => $otopo->id), $sort='allowsubmissionfromdate');

    foreach($sessions_list as $session) {
        $sessions[$session->id] = $session;
    }

    return $sessions;
}

function get_last_session_closed($otopo) {
    global $DB;

    $date = new DateTime();

    return $DB->get_record_sql('SELECT * FROM {otopo_session} WHERE otopo = :otopo AND allowsubmissiontodate < :now ORDER BY allowsubmissionfromdate DESC LIMIT 1', array('otopo' => $otopo->id, 'now' => $date->getTimestamp()));
}

function get_sessions_opened($otopo) {
    global $DB;

    $sessions = array();

    $date = new DateTime();
    $sessions_list = $DB->get_records_sql('SELECT * FROM {otopo_session} WHERE otopo = :otopo AND allowsubmissionfromdate < :now1 AND allowsubmissiontodate > :now2 ORDER BY allowsubmissionfromdate', array('otopo' => $otopo->id, 'now1' => $date->getTimestamp(), 'now2' => $date->getTimestamp()));

    foreach($sessions_list as $session) {
        $sessions[$session->id] = $session;
    }

    return $sessions;
}

function get_current_session($otopo, $user) {
    global $DB;

    $last_valid_session = $DB->get_record_sql('SELECT * FROM {otopo_user_valid_session} WHERE userid = :user AND otopo = :otopo AND session ' . ($otopo->session ? '>' : '<') . ' 0 ORDER BY id DESC LIMIT 1', array('user' => $user->id, 'otopo' => $otopo->id));

    if(!$last_valid_session) {
        $last_valid_session = 0;
    }
    else {
        $last_valid_session = $last_valid_session->session;
    }

    function endc( $array ) { return end( $array ); }
    if($otopo->session) {
        $sessions_o = get_sessions_opened($otopo);
        $sessions = array_keys($sessions_o);
        $last_session_id = endc($sessions);
        if($last_valid_session) {
            if($last_valid_session != $last_session_id) {
                $i = 1;
                foreach($sessions as $id) {
                    if($id == $last_valid_session) {
                        return array($i, next($sessions), $sessions_o[$id]);
                    }
                    ++$i;
                }
            }
        }
        else if (!empty($sessions)) {
            return array(1, $sessions[0], $sessions_o[$sessions[0]]);
        }
    }
    else {
        if(abs($last_valid_session) + 1 <= $otopo->limit_sessions)
            return array($last_valid_session - 1, $last_valid_session - 1);
    }

    return null;
}

function session_is_valid($otopo_id, $user, $session) {
    global $DB;

    return $DB->record_exists('otopo_user_valid_session', array('userid' => $user->id, 'otopo' => $otopo_id, 'session' => $session));
}

function session_is_closed($session) {
    global $DB;

    if($session > 0) {
        $date = new DateTime();
        return $DB->record_exists_sql('SELECT * FROM {otopo_session} WHERE id = :id AND (allowsubmissiontodate < :now1 OR allowsubmissionfromdate > :now2)', array('id' => $session, 'now1' => $date->getTimestamp(), 'now2' => $date->getTimestamp()));
    }
    return null;
}

function session_is_valid_or_closed($otopo_id, $user, $session) {
    global $DB;

    $valid = session_is_valid($otopo_id, $user, $session);

    $old = false;

    if($session > 0) {
        $old_or_future = session_is_closed($session);

        return $valid || $old_or_future;
    }
    else if($session < -1 && !$valid) {
        return !session_is_valid($otopo_id, $user, $session + 1);
    }

    return $valid;
}

function last_modification_on_session($otopo, $user, $session)
{
    global $DB;

    $max = $DB->get_record_sql('SELECT MAX(lastmodificationdate) AS lastmodificationdate FROM {otopo_user_otopo} INNER JOIN {otopo_item} AS it ON item = it.id WHERE userid = :user AND session = :session AND it.otopo = :otopo', array('user' => intval($user->id), 'session' => $session, 'otopo' => $otopo->id));

    if ($max && property_exists($max, 'lastmodificationdate')) {
        return $max->lastmodificationdate;
    }

    return null;
}

function last_modification($otopo, $user)
{
    global $DB;

    if ($otopo->session) {
        $max = $DB->get_record_sql('SELECT MAX(lastmodificationdate) AS lastmodificationdate FROM {otopo_user_otopo} INNER JOIN {otopo_item} AS it ON item = it.id WHERE userid = :user AND it.otopo = :otopo AND session > 0', array('user' => intval($user->id), 'otopo' => $otopo->id));
    } else {
        $max = $DB->get_record_sql('SELECT MAX(lastmodificationdate) FROM {otopo_user_otopo} INNER JOIN {otopo_item} AS it ON item = it.id WHERE userid = :user AND it.otopo = :otopo AND session < 0', array('user' => intval($user->id), 'otopo' => $otopo->id));
    }

    if ($max && property_exists($max, 'lastmodificationdate')) {
        return $max->lastmodificationdate;
    }

    return null;
}

function csv_from_items($items, $filename) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '";');

    $f = fopen('php://output', 'w');

    $max_degrees = 0;
    foreach ($items as $item) {
        if(count($item->degrees) > $max_degrees) {
            $max_degrees = count($item->degrees);
        }
    }

    $header = ['name', 'ord'];
    for($i = 0; $i < $max_degrees; $i++) {
        $header[] = 'degree' . ($i + 1) . '_name';
        $header[] = 'degree' . ($i + 1) . '_description';
        $header[] = 'degree' . ($i + 1) . '_grade';
        $header[] = 'degree' . ($i + 1) . '_ord';
    }
    fputcsv($f, $header, ',');

    foreach ($items as $item) {
        $it = (array)$item;
        $degrees = $it['degrees'];
        unset($it['id']);
        unset($it['degrees']);
        unset($it['color']);
        foreach (array_values($degrees) as $key => $degree) {
            unset($degree->id);
            foreach($degree as $field => $value) {
                $it['degree' . $key . '_' . $field] = $value;
            }
        }
        fputcsv($f, $it, ',');
    }

    fclose($f);
}

function prepare_data($o, &$items, &$otopos, $user) {
    global $DB;

    if($o->session) {
        $sessions = get_sessions($o);
    }
    else {
        $sessions = [];
        $max_sessions = 0;
        //foreach($items as $item) {
        //    if(array_key_exists($item->id, $otopos) && count($otopos[$item->id]) > $max_sessions) {
        //        $max_sessions = count($otopos[$item->id]);
        //    }
        //}
        $s = -1;
        while(session_is_valid($o->id, $user, $s)) {
            ++$max_sessions;
            --$s;
        }
        $colors = [
            "#323e70",
            "#684780",
            "#9b4e85",
            "#c7587e",
            "#e96d6e",
            "#fb8c5b",
            "#fdb14c",
            "#eed94e",
        ];
        $j = 0;
        if(session_is_valid($o->id, $user, -$max_sessions)) {
            $max_sessions += 1;
        }
        if($max_sessions == 0 && count($otopos) > 0) {
            $max_sessions += 1;
        }
        for($i=1; $i<=$max_sessions; $i++) {
            $sessions[] = (object)array('id' => -$i, 'color' => $colors[$j]);
            ++$j;
            if($j > 7)
                $j = 0;
        }
    }

    $i = 1;
    foreach($items as $item) {
        $item->index = $i;
        foreach($sessions as $session) {
            if(array_key_exists($item->id, $otopos) && array_key_exists($session->id, $otopos[$item->id])) {
                $otopo = &$otopos[$item->id][$session->id];
                $rank = 0;
                foreach($item->degrees as $key => $degree) {
                    if($degree->id == $otopo->degree) {
                        $rank = $key + 1;
                        $otopo->degree = $degree;
                        break;
                    }
                }
                $otopo->rank = $rank;
            }
        }
        ++$i;
    }
    $i = 1;
    foreach($sessions as $session) {
        $session->is_valid_or_closed = session_is_valid_or_closed($o->id, $user, $session->id);
        $session->is_valid = session_is_valid($o->id, $user, $session->id);
        $grader = $DB->get_record('otopo_grader', array('userid' => $user->id, 'session' => $session->id, 'otopo' => $o->id));
        $session->index = $i;
        if($grader) {
            $session->grade = $grader->grade;
            $session->comment = $grader->comment;
        }
        else {
            if($o->grade > 0) {
                $session->grade = calculate_grade($otopos, $items, $session, $o->grade);
                $session->comment = "";
            }
        }
        if($i == count($sessions)) {
            $session->last = true;
        }
        else {
            $session->last = false;
        }
        ++$i;
    }

    return $sessions;
}

function calculate_grade($otopos, $items, $session, $grade_max) {
    $num = 0;
    $den = 0;
    foreach($items as $item) {
        if(array_key_exists($item->id, $otopos) && array_key_exists($session->id, $otopos[$item->id])) {
            $otopo = $otopos[$item->id][$session->id];
            $num += $otopo->degree->grade;
        }
        $den += $item->degrees[count($item->degrees) - 1]->grade;

    }
    if($den == 0) {
        return null;
    }
    return intval($num / $den * $grade_max);
}


function get_my_evolution($otopo, $items, $items_sorted, $sessions, $otopos, $user, $visual = 'radar', $item = null) {
    $data = (object)array();

    $nbr_degrees_max = 0;
    foreach($items as $it) {
        if(count($it->degrees) > $nbr_degrees_max) {
            $nbr_degrees_max = count($it->degrees);
        }
    }
    $data->max = $nbr_degrees_max;

    if(!$item) {
        $charts = [];
        foreach ($sessions as $session) {
            if ($visual == 'radar') {
                $charts[$session->index] = array(
                    'id' => $session->index,
                    'grade' => $session->grade,
                    'comment' => $session->comment,
                    'label' => get_string('fill_autoeval', 'otopo') . ' ' . $session->index,
                    'labels' => [],
                    'fullLabel' => '',
                    'fullLabels' => [],
                    'color' => $session->color,
                    'allowsubmissionfromdate' => property_exists($session, 'allowsubmissionfromdate') && $session->allowsubmissionfromdate ? userdate($session->allowsubmissionfromdate, get_string('strftimedatetimeshort', 'core_langconfig')) : null,
                    'allowsubmissiontodate' => property_exists($session, 'allowsubmissiontodate') && $session->allowsubmissiontodate ? userdate($session->allowsubmissiontodate, get_string('strftimedatetimeshort', 'core_langconfig')) : null,
                    'datasets' => [[
                        'data' => [],
                        'labels' => [],
                        'label' => get_string('fill_autoeval', 'otopo') . ' ' . $session->index,
                        'backgroundColor' => [$session->color . 'dd'],
                        'hoverBackgroundColor' => [$session->color . '82'],
                        'borderColor' => [$session->color],
                        'borderWidth' => 1
                    ]]);
            } else {
                $charts[$session->index] = array(
                    'id' => $session->index,
                    'grade' => $session->grade,
                    'comment' => $session->comment,
                    'labels' => [],
                    'label' => get_string('fill_autoeval', 'otopo') . ' ' . $session->index,
                    'fullLabels' => [],
                    'color' => $session->color,
                    'datasets' => [[
                        'data' => [],
                        'labels' => [],
                        'fullLabels' => [],
                        'label' => get_string('fill_autoeval', 'otopo') . ' ' . $session->index,
                        'backgroundColor' => [],
                        'hoverBackgroundColor' => [],
                        'borderColor' => [],
                        'borderWidth' => 1
                    ]]);
            }
            foreach($items_sorted as $item) {
                $charts[$session->index]['labels'][] = strlen($item->name) > 26 ? (mb_substr($item->name, 0, 23) . '...') : $item->name;
                $charts[$session->index]['fullLabels'][] = $item->name;
                if ($visual == 'bar') {
                    $charts[$session->index]['datasets'][0]['backgroundColor'][] = $item->color . 'dd';
                    $charts[$session->index]['datasets'][0]['hoverBackgroundColor'][] = $item->color . '82';
                    $charts[$session->index]['datasets'][0]['borderColor'][] = $item->color;
                }
                if (array_key_exists($item->id, $otopos) && array_key_exists($session->id, $otopos[$item->id])) {
                    $charts[$session->index]['datasets'][0]['data'][] = $otopos[$item->id][$session->id]->rank;
                    $charts[$session->index]['datasets'][0]['labels'][] = $otopos[$item->id][$session->id]->degree->name;
                    $charts[$session->index]['datasets'][0]['fullLabels'][] = $item->name;
                } else {
                    $charts[$session->index]['datasets'][0]['data'][] = 0;
                    $charts[$session->index]['datasets'][0]['labels'][] = '';
                }
            }
        }

        foreach($sessions as $session) {
            $to_delete = true;
            foreach($charts[$session->index]['datasets'][0]['data'] as $value) {
                if ($value != 0) {
                    $to_delete = false;
                }
            }
            if ($to_delete) {
                unset($charts[$session->index]);
            }
        }

        $data->charts = array_values($charts);
    } else {
        $item = $items[$item];
        foreach ($items as $it) {
            if ($it->id == $item->id) {
                $item->index = $item->id;
                break;
            }
        }
        if ($visual == 'radar') {
            $chart_item = array(
                'id' => $item->id,
                'label' => $item->name,
                'labels' => [],
                'fullLabels' => [],
                'color' => $item->color,
                'datasets' => [[
                    'data' => [],
                    'labels' => [],
                    'label' => $item->name,
                    'backgroundColor' => [$item->color . 'dd'],
                    'hoverBackgroundColor' => [$item->color . '82'],
                    'borderColor' => [$item->color],
                    'borderWidth' => 1,
                ]]);
        } else {
            $chart_item = array(
                'id' => $item->id,
                'labels' => [],
                'label' => $item->name,
                'fullLabels' => [],
                'color' => $item->color,
                'datasets' => [[
                    'data' => [],
                    'labels' => [],
                    'label' => $item->name,
                    'backgroundColor' => [],
                    'hoverBackgroundColor' => [],
                    'borderColor' => [], 
                    'borderWidth' => 1
                ]]);
        }
        $data->chart_item_degrees = [];
        foreach($sessions as $session) {
            //On n'ajoute pas la session en cours
            if (!$session->is_valid_or_closed) {
                continue;
            }
            $chart_item['labels'][] = $item->name;
            $chart_item['fullLabels'][] = $otopos[$item->id][$session->id]->degree->name;
            if ($visual == 'bar') {
                $chart_item['datasets'][0]['backgroundColor'][] = $session->color . 'dd';
                $chart_item['datasets'][0]['hoverBackgroundColor'][] = $session->color . '82';
                $chart_item['datasets'][0]['borderColor'][] = $session->color;
                $chart_item['datasets'][0]['label'] = $otopos[$item->id][$session->id]->degree->name;
            }
            if (array_key_exists($item->id, $otopos) && array_key_exists($session->id, $otopos[$item->id])) {
                $chart_item['datasets'][0]['data'][] = $otopos[$item->id][$session->id]->rank;
                $data->chart_item_degrees[] = [
                    'label' => $otopos[$item->id][$session->id]->degree->name,
                    'degree' => $otopos[$item->id][$session->id]->degree,
                    'color' => $session->color, 'rank' => $otopos[$item->id][$session->id]->rank,
                    'allowsubmissionfromdate' => property_exists($session, 'allowsubmissionfromdate') ? userdate($session->allowsubmissionfromdate, get_string('strftimedatetimeshort', 'core_langconfig')) : null,
                    'allowsubmissiontodate' => property_exists($session, 'allowsubmissiontodate') ? userdate($session->allowsubmissiontodate, get_string('strftimedatetimeshort', 'core_langconfig')) : null,
                    'lastmodificationonsession' => userdate(last_modification_on_session($otopo, $user, $session->id), get_string('strftimedatetimeshort', 'core_langconfig'))
                ];
            } else {
                $chart_item['datasets'][0]['data'][] = 0;
            }
        }
        $data->chart_item = $chart_item;
    }

    return $data;
}

function parse_csv ($csv_string, $delimiter = ",", $skip_empty_lines = true, $trim_fields = true)
{
    $enc = preg_replace('/(?<!")""/', '!!Q!!', $csv_string);
    $enc = preg_replace_callback(
        '/"(.*?)"/s',
        function ($field) {
            return urlencode(utf8_encode($field[1]));
        },
        $enc
    );
    $lines = preg_split($skip_empty_lines ? ($trim_fields ? '/( *\R)+/s' : '/\R+/s') : '/\R/s', $enc);
    return array_map(
        function ($line) use ($delimiter, $trim_fields) {
            $fields = $trim_fields ? array_map('trim', explode($delimiter, $line)) : explode($delimiter, $line);
            return array_map(
                function ($field) {
                    return str_replace('!!Q!!', '"', utf8_decode(urldecode($field)));
                },
                $fields
            );
        },
        $lines
    );
}

function is_open($otopo) {
    $time = time();
    return ($otopo->allowsubmissionfromdate == '0' || $otopo->allowsubmissionfromdate <= $time)
                                                  && ($otopo->allowsubmissiontodate == '0' || $otopo->allowsubmissiontodate >= $time);
}
?>
