<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Prints an instance of mod_otopo.
 *
 * @package     mod_otopo
 * @copyright   2022 Kosmos <moodle@kosmos.fr>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

global $CFG;

require(__DIR__.'/../../config.php');
require_once(__DIR__.'/lib.php');
require_once(__DIR__.'/sessionsform.php');
require_once(__DIR__.'/fromtemplateform.php');
require_once(__DIR__.'/importgridform.php');
require_once(__DIR__.'/templateform.php');
require_once(__DIR__.'/mod_form.php');
require_once(__DIR__.'/locallib.php');
require_once($CFG->dirroot . '/course/modlib.php');
require_once($CFG->libdir . "/xsendfilelib.php");
require_once($CFG->dirroot.'/calendar/lib.php');

use core_table\local\filter\filter;
use core_table\local\filter\integer_filter;
use core_table\local\filter\string_filter;
use mod_otopo\output\grading_app;

define('DEFAULT_PAGE_SIZE', 20);
define('SHOW_ALL_PAGE_SIZE', 5000);


// Activity instance id.
$o = optional_param('o', 0, PARAM_INT);

if($o == 0) {
    // Course module id.
    $id = optional_param('id', 0, PARAM_INT);
}
else {
    $id = 0;
}

if ($id) {
    $cm = get_coursemodule_from_id('otopo', $id, 0, false, MUST_EXIST);
    $course = $DB->get_record('course', array('id' => $cm->course), '*', MUST_EXIST);
    $moduleinstance = $DB->get_record('otopo', array('id' => $cm->instance), '*', MUST_EXIST);
} else {
    $moduleinstance = $DB->get_record('otopo', array('id' => $o), '*', MUST_EXIST);
    $course = $DB->get_record('course', array('id' => $moduleinstance->course), '*', MUST_EXIST);
    $cm = get_coursemodule_from_instance('otopo', $moduleinstance->id, $course->id, false, MUST_EXIST);
}
$perpage = optional_param('perpage', DEFAULT_PAGE_SIZE, PARAM_INT); // How many per page.

require_login($course, true, $cm);

$modulecontext = context_module::instance($cm->id);
$coursecontext = context_course::instance($course->id);

require_capability('mod/otopo:view', $modulecontext);

$can_admin = has_capability('mod/otopo:admin', $modulecontext);
$can_grade = has_capability('mod/otopo:grade', $modulecontext);
$can_export_results = has_capability('mod/otopo:export_results', $modulecontext);
$can_managetemplates = has_capability('mod/otopo:managetemplates', $modulecontext);


// Action
$action = optional_param('action', '', PARAM_TEXT);

// Object
$object = optional_param('object', 'params', PARAM_TEXT);

$sectionreturn = optional_param('sr', null, PARAM_INT);

$PAGE->set_url('/mod/otopo/view.php', array('id' => $cm->id, 'action' => $action, 'object' => $object));
$PAGE->set_title(format_string($moduleinstance->name));
$PAGE->set_heading(format_string($course->fullname));
$PAGE->set_context($modulecontext);

$output = $PAGE->get_renderer('mod_otopo');

$body = "";

$event = \mod_otopo\event\activity_viewed::create(array('context' => $modulecontext));
$event->trigger();

if($can_admin || $can_grade || $can_export_results) {
    $content = NULL;

    if($action == '')
        $action = 'edit';

    if($action == 'grader') {
        require_capability('mod/otopo:grade', $modulecontext);

        $PAGE->set_pagelayout('embedded');

        $userid = optional_param('user', 0, PARAM_INT);
        $session = optional_param('session', null, PARAM_INT);

        $framegrader = new grading_app($userid, $course, $cm, $moduleinstance, $modulecontext, $session);

        echo $output->header();
        echo $output->render($framegrader);
        echo $output->footer();

        return;
    }

    if($can_admin) {
        if($action == 'edit') {
            if($object == "sessions") {
                include('vue.php');

                $session_delete = optional_param('session-delete', -1, PARAM_INT);

                $PAGE->requires->js_call_amd('mod_otopo/sessions','initDeleteSession', array('wwwroot' => $CFG->wwwroot));
                $PAGE->requires->js_call_amd('mod_otopo/sessions','initColorPicker');

                $sessions = $DB->get_records('otopo_session', array('otopo' => $moduleinstance->id), $sort='allowsubmissionfromdate', $fields='*');

                $toform = toform_from_sessions($sessions);

                if($session_delete >= 0) {
                    if($toform && array_key_exists($session_delete, $toform->id)) {
                        $session = $DB->get_record('otopo_session', array('id' => $toform->id[$session_delete]));
                        if($session->event_start) {
                            $event = calendar_event::load($session->event_start);
                            if($event) {
                                $event->delete();
                            }
                        }
                        if($session->event_end) {
                            $event = calendar_event::load($session->event_end);
                            if($event) {
                                $event->delete();
                            }
                        }

                        $DB->delete_records('otopo_session', array('id' => $toform->id[$session_delete]));
                        $sessions = $DB->get_records('otopo_session', array('otopo' => $moduleinstance->id), $sort='allowsubmissionfromdate', $fields='*');

                        $items = $DB->get_records('otopo_item', array('otopo' => $moduleinstance->id));
                        if(!empty($items)) {
                            list($insql, $params) = $DB->get_in_or_equal(array_keys($items));
                            $params[] = $toform->id[$session_delete];

                            $DB->delete_records_select('otopo_user_otopo', "item $insql AND session = ?", $params);
                            $DB->delete_records('otopo_user_valid_session', array('otopo' => $moduleinstance->id, 'session' => $toform->id[$session_delete]));
                        }

                        $toform = toform_from_sessions($sessions);
                        $moduleinstance->instance = $moduleinstance->id;
                        otopo_update_instance($moduleinstance);
                    }
                }

                $mform = new sessions_form(null, array('o' => $moduleinstance->id, 'sessions' => $moduleinstance->sessions, 'count_sessions' => count($sessions)));

                if ($mform->is_cancelled()) {
                    //Handle form cancel operation, if cancel button is present on form
                    redirect(new moodle_url('/mod/otopo/view.php', array('o' => $moduleinstance->id, 'action' => $action, 'object' => $object)));
                }
                else if ($fromform = $mform->get_data()) {
                    //In this case you process validated data. $mform->get_data() returns data posted in form.

                    for($i=0; $i<$fromform->option_repeats; $i++) {
                        $session = new stdClass();
                        $session->otopo = intval($moduleinstance->id);
                        $session->name = $fromform->name[$i];
                        $session->color = $fromform->color[$i];
                        $session->allowsubmissionfromdate = $fromform->allowsubmissionfromdate[$i];
                        $session->allowsubmissiontodate = $fromform->allowsubmissiontodate[$i];
                        if($fromform->id[$i] != 0) {
                            $session->id = $fromform->id[$i];
                            $DB->update_record('otopo_session', $session);
                            $session = $DB->get_record('otopo_session', array('id' => $toform->id[$i]));

                            if($session->event_start) {
                                $event = calendar_event::load($session->event_start);
                                if($event) {
                                    $event->name = $moduleinstance->name . ' - ' . $session->name . ' - ' . get_string('start', 'otopo');
                                    $event->timestart = $session->allowsubmissionfromdate;
                                    $event->timesort = $session->allowsubmissionfromdate;
                                    $event->visible = instance_is_visible('otopo', $moduleinstance);
                                    $event->update($event, false);
                                }
                            }
                            if($session->event_end) {
                                $event = calendar_event::load($session->event_end);
                                if($event) {
                                    $event->name = $moduleinstance->name . ' - ' . $session->name . ' - ' . get_string('end', 'otopo');
                                    $event->timeend = $session->allowsubmissiontodate;
                                    $event->timesort = $session->allowsubmissiontodate;
                                    $event->visible = instance_is_visible('otopo', $moduleinstance);
                                    $event->update($event, false);
                                }
                            }
                        }
                        else {
                            $fromform->id = $DB->insert_record('otopo_session', $session);
                            $session->id = $fromform->id;

                            $event = new stdClass();
                            $event->eventtype = OTOPO_EVENT_TYPE_SESSION;
                            $event->type = CALENDAR_EVENT_TYPE_ACTION;
                            $event->name = $moduleinstance->name . ' - ' . $session->name . ' - ' . get_string('start', 'otopo');
                            $event->format = FORMAT_HTML;
                            $event->courseid = $moduleinstance->course;
                            $event->groupid = 0;
                            $event->userid = 0;
                            $event->modulename = 'otopo';
                            $event->instance = $moduleinstance->id;
                            $event->repeatid = $session->id;
                            $event->timestart = $session->allowsubmissionfromdate;
                            $event->timeduration = 0;
                            $event->timesort = $session->allowsubmissionfromdate;
                            $event->visible = instance_is_visible('otopo', $moduleinstance);

                            $event = calendar_event::create($event, false);

                            $session->event_start = $event->id;

                            $event = new stdClass();
                            $event->eventtype = OTOPO_EVENT_TYPE_SESSION;
                            $event->type = CALENDAR_EVENT_TYPE_ACTION;
                            $event->name = $moduleinstance->name . ' - ' . $session->name . ' - ' . get_string('end', 'otopo');
                            $event->format = FORMAT_HTML;
                            $event->courseid = $moduleinstance->course;
                            $event->groupid = 0;
                            $event->userid = 0;
                            $event->modulename = 'otopo';
                            $event->instance = $moduleinstance->id;
                            $event->repeatid = $session->id;
                            $event->timestart = $session->allowsubmissiontodate;
                            $event->timeduration = 0;
                            $event->timesort = $session->allowsubmissiontodate;
                            $event->visible = instance_is_visible('otopo', $moduleinstance);

                            $event = calendar_event::create($event, false);

                            $session->event_end = $event->id;

                            $DB->update_record('otopo_session', $session);
                        }
                    }

                    $moduleinstance->instance = $moduleinstance->id;
                    otopo_update_instance($moduleinstance);

                    redirect(new moodle_url('/mod/otopo/view.php', array('o' => $moduleinstance->id, 'action' => $action, 'object' => $object)));
                }
                else {
                    // this branch is executed if the form is submitted but the data doesn't validate and the form should be redisplayed
                    // or on the first display of the form.

                    //Set default data (if any)
                    $mform->set_data($toform);
                }

                $content = $output->heading(get_string('sessions_settings', 'otopo')). $mform->render();
            }
            else if ($object == 'grids') {
                include('vue.php');

                $PAGE->requires->js_call_amd('mod_otopo/grids','initGrid', array($moduleinstance->id, has_otopo($moduleinstance->id)));

                $content = $OUTPUT->render_from_template('mod_otopo/grids', array('cm' => $cm, 'can_managetemplates' => $can_managetemplates, 'has_otopo' => has_otopo($moduleinstance->id)));
            }
            else if ($object == 'templates') {
                require_capability('mod/otopo:managetemplates', $modulecontext);

                $items = get_items_sorted_from_otopo($moduleinstance->id);

                $nbr_degrees_max = table_items($items);

                $template = $OUTPUT->render_from_template('mod_otopo/grids_table', array('items' => array_values($items), 'nbr_degrees_max' => $nbr_degrees_max));

                $mform = new template_form(null, array('o' => $moduleinstance->id, 'template' => $template, 'action' => 'edit'));

                if ($mform->is_cancelled()) {
                    redirect(new moodle_url('/mod/otopo/view.php', array('id' => $cm->id, 'action' => 'preview', 'object' => 'grids')));
                }
                else if ($fromform = $mform->get_data()) {
                    $template = (object)array('id' => null, 'name' => $fromform->name);
                    $template->id = $DB->insert_record('otopo_template', $template);
            
                    copy_items(-$template->id, $items);

                    redirect(new moodle_url('/mod/otopo/templates.php', array('id' => $template->id, 'cmid' => $cm->id)));
                }

                $content = $mform->render();
            }
        }
        else if ($action == 'preview') {
            $items = get_items_sorted_from_otopo($moduleinstance->id);
            if ($object == 'grids') {
                $nbr_degrees_max = table_items($items);

                $content = $OUTPUT->render_from_template('mod_otopo/preview_grids', array('cm' => $cm,'items' => array_values($items), 'nbr_degrees_max' => $nbr_degrees_max, 'can_managetemplates' => $can_managetemplates));
            }
            else if ($object == 'sessions') {
                include('vue.php');

                $args = array(
                    $moduleinstance->id,
                    (boolean)$moduleinstance->showteachercomments,
                    $OUTPUT->image_url('star', 'mod_otopo')->out(),
                    $OUTPUT->image_url('star_container', 'mod_otopo')->out(),
                    $OUTPUT->image_url('help', 'mod_otopo')->out(),
                    $OUTPUT->image_url('plus', 'mod_otopo')->out(),
                    $OUTPUT->image_url('minus', 'mod_otopo')->out(),
                );

                $PAGE->requires->js_call_amd('mod_otopo/preview_sessions','init', $args);

                $content = $OUTPUT->render_from_template('mod_otopo/preview_sessions', array());
            }
        }
        else if ($action == 'export') {
            if ($object == 'grids') {
                
                $items = get_items_sorted_from_otopo($moduleinstance->id);

                csv_from_items($items, 'grids_' . $moduleinstance->id . '.csv');

                return;
            }
            else if ($object == 'group') {
                require_capability('mod/otopo:export_results', $modulecontext);

                $users = optional_param('users', 'params', PARAM_TEXT);
                if(!empty($users)) {
                    $users = explode(',', $users);
                }
                else {
                    $users = [];
                }

                $distribution = get_distribution_by_item($moduleinstance, $users);
                $items = get_items_sorted_from_otopo($moduleinstance->id);

                header('Content-Type: text/csv');
                header('Content-Disposition: attachment; filename="group_exports_' . $moduleinstance->id . '.csv";');

                $f = fopen('php://output', 'w');

                $nbr_degrees_max = 0;
                foreach ($items as $item) {
                    if(count($item->degrees) > $nbr_degrees_max) {
                        $nbr_degrees_max = count($item->degrees);
                    }
                }

                $header = ['degree'];
                foreach(array_values($items) as $key1 => $item) {
                    $header[] = 'item' . ($key1 + 1);
                }
                fputcsv($f, $header, ',');

                for($i = 0; $i < $nbr_degrees_max; $i++) {
                    $row = ['degree' => 'degree'. ($i + 1)];
                    foreach(array_values($items) as $key1 => $item) {
                        if(array_key_exists($i, $distribution) && array_key_exists($key1, $distribution[$i])) {
                            $row['item' . ($key1 + 1)] = $distribution[$i][$key1];
                        }
                        else {
                            $row['item' . ($key1 + 1)] = '';
                        }
                    }
                    fputcsv($f, $row, ',');
                }

                fclose($f);

                return;
            }
            else if ($object == 'individual') {
                require_capability('mod/otopo:export_results', $modulecontext);

                $sessions = optional_param('sessions', 'params', PARAM_TEXT);
                if(!empty($sessions)) {
                    $sessionsParam = explode(',', $sessions);
                    $sessions = [];
                    foreach($sessionsParam as $param) {
                        $arr = explode('_', $param);
                        if(count($arr) == 2) {
                            $user = $arr[0];
                            $session = $arr[1];
                            if(!array_key_exists($user, $sessions)) {
                                $sessions[$user] = [];
                            }
                            $sessions[$user][] = $session;
                        }
                    }
                }
                else {
                    $sessions = [];
                }

                $users_sessions_with_otopos = get_users_sessions_with_otopos($moduleinstance);
                $graders = get_graders($moduleinstance);

                header('Content-Type: text/csv');
                header('Content-Disposition: attachment; filename="individual_exports_' . $moduleinstance->id . '.csv";');

                $f = fopen('php://output', 'w');


                $header = ['fullname', 'session', 'grade', 'comment'];
                fputcsv($f, $header, ',');

                if($moduleinstance->session) {
                    $sessions_ids = get_sessions($moduleinstance);
                }
                foreach($sessions as $user => $user_sessions) {
                    if(array_key_exists($user, $users_sessions_with_otopos)) {
                            $userObj = $DB->get_record('user', array('id' => $user));
                            $fullname = fullname($userObj);
                            foreach($user_sessions as $session) {
                                if(array_key_exists($session, $users_sessions_with_otopos[$user])) {
                                        if($moduleinstance->session) {
                                            $i = 1;
                                            foreach(array_keys($sessions_ids) as $id) {
                                                if($id == $session) {
                                                    $id_abs = $i;
                                                    break;
                                                }
                                                ++$i;
                                            }
                                        }
                                        else {
                                            $id_abs = abs($session);
                                        }
                                        if(array_key_exists($user, $graders)
                                           && array_key_exists($session, $graders[$user])
                                           && $graders[$user][$session]->grade) {
                                            $grade = $graders[$user][$session]->grade;
                                        }
                                        else {
                                            $grade = '';
                                        }
                                        if(array_key_exists($user, $graders)
                                           && array_key_exists($session, $graders[$user])
                                           && $graders[$user][$session]->comment) {
                                            $comment = $graders[$user][$session]->comment;
                                        }
                                        else {
                                            $comment = '';
                                        }
                                        $row = [
                                            'fullname' => $fullname,
                                            'session' => get_string('session', 'otopo') . ' ' . $id_abs,
                                            'grade' => $grade,
                                            'comment' => strip_tags($comment),
                                        ];
                                        fputcsv($f, $row, ',');
                                    }
                                    }
                            }
                }

                fclose($f);

                return;
            }
        }
        else if ($action == 'import') {
            if(!has_otopo($moduleinstance->id)) {
                if ($object == 'grids') {
                    $mform = new importgrid_form(null, array('o' => $moduleinstance->id));
                    if ($mform->is_cancelled()) {
                        redirect(new moodle_url('/mod/otopo/view.php', array('id' => $cm->id, 'action' => 'edit', 'object' => 'grids')));
                    }
                    else if ($fromform = $mform->get_data()) {
                        $content = $mform->get_file_content('csv');

                        $csv = parse_csv($content);

                        if(!empty($csv)) {
                            $header = array_shift($csv);
                            unset($csv[count($csv)-1]);

                            foreach($csv as $i => $line) {
                                $item = [];
                                $degrees = [];
                                foreach($line as $j => $value) {
                                    if(substr($header[$j], 0, 6) == 'degree') {
                                        $matches = [];
                                        preg_match('/^degree(\d+)_(\w*)$/', $header[$j], $matches);
                                        $k = intval($matches[1]);
                                        $field = $matches[2];
                                        if(array_key_exists($k, $degrees)) {
                                            $degrees[$k-1] = [];
                                        }
                                        $degrees[$k-1][$field] = $value;
                                    }
                                    else {
                                        $item[$header[$j]] = $value;
                                    }

                                }
                                if(!empty($degrees)) {
                                    $item['degrees'] = $degrees;
                                }
                                $items[] = $item;
                            }

                            if(!empty($items)) {
                                delete_items($moduleinstance->id);

                                foreach($items as $item) {
                                    $new_item = ['name' => $item['name'], 'ord' => $item['ord'], 'otopo' => $moduleinstance->id];
                                    $item_id = $DB->insert_record('otopo_item', $new_item);
                                    foreach($item['degrees'] as $degree) {
                                        $new_degree = [
                                            'name' => $degree['name'],
                                            'description' => $degree['description'],
                                            'grade' => intval($degree['grade']),
                                            'ord' => intval($degree['ord']),
                                            'item' => $item_id
                                        ];
                                        $DB->insert_record('otopo_item_degree', $new_degree);
                                    }
                                }
                            }

                            redirect(new moodle_url('/mod/otopo/view.php', array('id' => $cm->id, 'action' => 'edit', 'object' => 'grids')));
                        }
                    }
                    $content = $mform->render();
                }
                else if ($object == 'templates') {
                    $templates = get_templates();

                    $mform = new fromtemplate_form(null, array('templates' => $templates, 'id' => $cm->id, 'action' => 'edit'));
                    if ($mform->is_cancelled()) {
                        redirect(new moodle_url('/mod/otopo/view.php', array('id' => $cm->id, 'action' => 'edit', 'object' => 'grids')));
                    }
                    else if ($fromform = $mform->get_data()) {
                        $template_id = intval($fromform->template);

                        $items = get_items_from_otopo(-$template_id);

                        delete_items($moduleinstance->id);
                        copy_items($moduleinstance->id, $items);

                        redirect(new moodle_url('/mod/otopo/view.php', array('id' => $cm->id, 'action' => 'edit', 'object' => 'grids')));
                    }

                    $content = $mform->render();
                }
            }
        }
        else if ($action == 'report') {
            $filterset = new \mod_otopo\table\participants_filterset();
            if($object == 'group') {
                $filterset = new \mod_otopo\table\participants_no_row_filterset();
                $filterset->add_filter(new integer_filter('session', filter::JOINTYPE_DEFAULT, [(int)0]));
            }
            $filterset->add_filter(new integer_filter('otopo', filter::JOINTYPE_DEFAULT, [(int)$moduleinstance->id]));
            $filterset->add_filter(new integer_filter('courseid', filter::JOINTYPE_DEFAULT, [(int)$course->id]));

            if ($object == 'individual') {
                $participanttable = new \mod_otopo\table\participants("user-index-participants-{$course->id}");
            } else if ($object == 'group') {
                $participanttable = new \mod_otopo\table\participants_no_row("user-index-participants-{$course->id}");
            }
            $participanttable->set_action($action);
            $participanttable->set_object($object);

            // Render the user filters.
            $content .= '<div class="no-print">';
            $renderable = new \core_user\output\participants_filter($coursecontext, $participanttable->uniqueid);
            $templatecontext = $renderable->export_for_template($OUTPUT);
            $templatecontext->otopo = $moduleinstance;
            if ($object == 'group') {
                $templatecontext->session_filter = true;
                $templatecontext->sessions = array_values(get_sessions($moduleinstance));
                $templatecontext->has_sessions = !empty($templatecontext->sessions);
            } else {
                $templatecontext->session_filter = false;
            }

            // Needed because of a breaking change in moodle user plugin in 4.1
            $moodleVersion = get_config('')->version;
            if ($moodleVersion < 2022112800) {
                $templatecontext->moodlePre41 = true;
            }

            $content .= $OUTPUT->render_from_template('mod_otopo/participantsfilter', $templatecontext);
            $content .= '</div>';

            $content .= '<div class="userlist">';

            // Do this so we can get the total number of rows.
            ob_start();
            $participanttable->set_filterset($filterset);
            $content .= $participanttable->out($perpage, true);
            $participanttablehtml = ob_get_contents();
            ob_end_clean();

            $content .=  html_writer::start_tag('form', [
                'action' => 'action_redir.php',
                'method' => 'post',
                'id' => 'participantsform',
                'data-course-id' => $course->id,
                'data-otopo' => $moduleinstance->id,
                'data-table-unique-id' => $participanttable->uniqueid,
                'data-table-default-per-page' => ($perpage < DEFAULT_PAGE_SIZE) ? $perpage : DEFAULT_PAGE_SIZE,
            ]);
            $content .= '<div>';
            $content .= '<input type="hidden" name="sesskey" value="'.sesskey().'" />';
            $content .= '<input type="hidden" name="returnto" value="'.s($PAGE->url->out(false)).'" />';

            $content .=  html_writer::tag(
                'p',
                get_string('countparticipantsfound', 'core_user', $participanttable->totalrows),
                [
                    'data-region' => 'participant-count',
                    'class' => 'no-print',
                ]
            );

            $content .= $participanttablehtml;

            $perpagesize = DEFAULT_PAGE_SIZE;
            $perpagevisible = false;
            $perpagestring = '';

            $perpageurl = new moodle_url('/mod/otopo/view.php', array('id' => $cm->id, 'action' => 'report', 'object' => 'individual'));

            if ($perpage == SHOW_ALL_PAGE_SIZE && $participanttable->totalrows > DEFAULT_PAGE_SIZE) {
                $perpageurl->param('perpage', $participanttable->totalrows);
                $perpagesize = SHOW_ALL_PAGE_SIZE;
                $perpagevisible = true;
                $perpagestring = get_string('showperpage', '', DEFAULT_PAGE_SIZE);
            } else if ($participanttable->get_page_size() < $participanttable->totalrows) {
                $perpageurl->param('perpage', SHOW_ALL_PAGE_SIZE);
                $perpagesize = SHOW_ALL_PAGE_SIZE;
                $perpagevisible = true;
                $perpagestring = get_string('showall', '', $participanttable->totalrows);
            }

            $perpageclasses = '';
            if (!$perpagevisible) {
                $perpageclasses = 'hidden';
            }
            $content .= $OUTPUT->container(html_writer::link(
                $perpageurl,
                $perpagestring,
                [
                    'data-action' => 'showcount',
                    'data-target-page-size' => $perpagesize,
                    'class' => $perpageclasses,
                ]
            ), [], 'showall');
            $bulkoptions = (object) [
                'uniqueid' => $participanttable->uniqueid,
            ];

            $content .= '</form>';

            if($object == 'individual') {
                $content .= '<div class="d-flex justify-content-end mt-3">';
                if(has_capability('mod/otopo:export_results', $modulecontext)) {
                    $content .= '<a class="btn-otopo ml-5 mr-2 no-print" href="#" onclick="window.print()">' . get_string('print', 'otopo') . '</a>';
                    $content .= '<a id="export-csv" class="btn-otopo ml-2 mr-5 no-print" href="#">'  . get_string('export_as_csv', 'otopo') . '</a>';
                }
                else {
                    $content .= '<a class="btn-otopo ml-5 mr-5 no-print" href="#" onclick="window.print()">' . get_string('print', 'otopo') . '</a>';
                }
                $content .= '</div>';
            }

            $PAGE->requires->js_call_amd('core_user/participants', 'init', [$bulkoptions]);
            if ($object == 'individual') {
                $PAGE->requires->js_call_amd('mod_otopo/report_individual', 'init', ['cmid' => $cm->id, 'wwwroot' => $CFG->wwwroot]);
            }
            $content .= '</div>';

            $content .= '</div>';

        }

        if(!$content) {
            $content = $OUTPUT->render_from_template('mod_otopo/params', array('cm' => $cm));
        }
    }


    $renderable = new view_page($cm, $moduleinstance, $action, $object, $content, $can_admin, $can_grade, $can_export_results, $can_managetemplates, has_otopo($moduleinstance->id));

    $body .= $output->render($renderable);
}

if(has_capability('mod/otopo:fill', $modulecontext)) {
    $session_valid = false;
    $session_valid_or_closed = false;
    $session_o = null;
    $session_key = 0;
    $session = 0;
    $mform = null;

    if(($action == 'edit' && $object == 'params') || $action == '')
        $action = 'progress';

    if($action == 'evaluate') {
        $session = optional_param('session', 0, PARAM_INT);
        if($session == 0) {
            $session = get_current_session($moduleinstance, $USER);
            $session_key = $session[0];
            if($session[1] > 0) {
                $session_o = $session[2];
            }
            else {
                $session_o = null;
            }
            $session = $session[1];
        }
        else {
            if($moduleinstance->session) {
                $sessions = get_sessions($moduleinstance);
                if(!array_key_exists($session, $sessions)) {
                    $session = 0;
                }
                else {
                    $session_o = $sessions[$session];
                    $i = 1;
                    foreach($sessions as $s) {
                        if($s->id == $session) {
                            $session_key = $i;
                            break;
                        }
                        ++$i;
                    }
                }
            }
            else {
                if(abs($session) <= $moduleinstance->limit_sessions) {
                    $session_key = abs($session);
                }
                else {
                    $session = 0;
                }
            }
        }
        $session_valid = session_is_valid($moduleinstance->id, $USER, $session);
        $session_valid_or_closed = session_is_valid_or_closed($moduleinstance->id, $USER, $session);
        $is_open = is_open($moduleinstance);

        include('vue.php');

        $args = array(
            $moduleinstance->id,
            (boolean)$moduleinstance->showteachercomments,
            $session,
            $session_valid_or_closed,
            $is_open,
            $OUTPUT->image_url('star', 'mod_otopo')->out(),
            $OUTPUT->image_url('star_container', 'mod_otopo')->out(),
            $OUTPUT->image_url('help', 'mod_otopo')->out(),
            $OUTPUT->image_url('plus', 'mod_otopo')->out(),
            $OUTPUT->image_url('minus', 'mod_otopo')->out(),
        );

        $PAGE->requires->js_call_amd('mod_otopo/evaluate','init', $args);
    }
    else if($action == 'validate') {
        $session = optional_param('session', 0, PARAM_INT);
        $session_valid = session_is_valid($moduleinstance->id, $USER, $session);
        if(!$session_valid) {
            $DB->insert_record('otopo_user_valid_session', array('userid' => $USER->id, 'session' => $session, 'otopo' => $moduleinstance->id));
        }
        otopo_update_grades($moduleinstance, $USER->id);

        $event = \mod_otopo\event\session_closed::create(array('context' => $modulecontext));
        $event->trigger();

        redirect(new moodle_url('/mod/otopo/view.php', array('id' => $cm->id, 'action' => 'progress')));
    }
    else if($action == 'evolution') {
        $PAGE->requires->js_call_amd('mod_otopo/evolution','init', array($moduleinstance->id, get_string('autoeval_degree', 'otopo')));
    }

    $renderable = new view_fill_page($cm, $moduleinstance, $action, $session, $session_key, $session_valid, $session_valid_or_closed, $session_o, $mform);

    $body .= $output->render($renderable);
}

echo $OUTPUT->header();

echo $body;

echo $OUTPUT->footer();
?>
