<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Defines renderers class
 *
 * @package     mod_otopo
 * @copyright   2022 Kosmos <moodle@kosmos.fr>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
require_once(__DIR__.'/locallib.php');

use mod_otopo\output\grading_app;

class view_page implements renderable, templatable {
    /** @var $cm Course module */
    var $cm = null;

    /** @var $otopo Otopo module instance */
    var $otopo = null;

    /** @var $action */
    var $action = 'edit';

    /** @var $content Content to insert */
    var $content = null;

    /** @var $can_admin Can admin capability */
    var $can_admin = null;

    /** @var $can_grade Can grade capability */
    var $can_grade = null;

    /** @var $can_export_results Can export results capability */
    var $can_export_results = null;

    /** @var $can_managetemplates Can export results capability */
    var $can_managetemplates = null;

    public function __construct($cm, $otopo, $action, $object, $content, $can_admin, $can_grade, $can_export_results, $can_managetemplates) {
        $this->action = $action;
        $this->object = $object;
        $this->cm = $cm;
        $this->otopo = $otopo;
        $this->content = $content;
        $this->can_admin = $can_admin;
        $this->can_grade = $can_grade;
        $this->can_export_results = $can_export_results;
        $this->can_managetemplates = $can_managetemplates;
    }

    /**
     * Export this data so it can be used as the context for a mustache template.
     *
     * @return stdClass
     */
    public function export_for_template(renderer_base $output) {
        $data = new stdClass();
        $data->edit = false;
        $data->preview = false;
        $data->sessions = false;
        $data->params = false;
        $data->grids = false;
        $data->templates = false;
        $data->report = false;
        $data->content = $this->content;
        $data->can_admin = $this->can_admin;
        $data->can_grade = $this->can_grade;
        $data->can_export_results = $this->can_export_results;
        $data->can_managetemplates = $this->can_managetemplates;
        $data->can_consult = $this->can_export_results || $this->can_grade;
        $data->disabled_sessions = !$this->otopo->session;
        switch($this->action) {
        case 'import':
        case 'create-from-template':
        case 'edit':
            $data->edit = true;
            break;
        case 'preview':
            $data->preview = true;
            break;
        case 'report':
            $data->report = true;
            break;
        }
        $data->action_ok = $data->edit || $data->preview || $data->report;
        switch($this->object) {
        case 'params':
            $data->params = true;
            break;
        case 'sessions':
            $data->sessions = true;
            break;
        case 'grids':
            $data->grids = true;
            break;
        case 'templates':
            if($this->action == 'import') {
                $data->grids = true;
            }
            else {
                $data->templates = true;
            }
            break;
        case 'individual':
            $data->individual = true;
            break;
        case 'group':
            $data->group = true;
            break;
        }
        $data->cm = $this->cm;
        $data->otopo = $this->otopo;
        return $data;
    }
}

class templates_page implements renderable, templatable {
    /** @var $template Otopo template */
    var $template = null;

    /** @var $action */
    var $action = 'show';

    /** @var $form to show */
    var $form = null;

    /** @var $cmid from module */
    var $cmid = null;

    public function __construct($template, $action, $form, $cmid) {
        $this->template = $template;
        $this->action = $action;
        $this->form = $form;
        $this->cmid = $cmid;
    }

    /**
     * Export this data so it can be used as the context for a mustache template.
     *
     * @return stdClass
     */
    public function export_for_template(renderer_base $output) {
        $data = new stdClass();
        $data->show = false;
        $data->edit = false;
        $data->create = false;
        $data->form = $this->form ? $this->form->render() : "";
        $data->template = $this->template;
        $data->templates = null;
        $data->items = null;
        $data->cmid = $this->cmid;
        if(!$this->template) {
            $data->templates = array_values(get_templates());
        }
        else {
            $items = get_items_sorted_from_otopo(-$this->template->id);
            $data->nbr_degrees_max = table_items($items);
            $data->items = array_values($items);
        }
        if($this->action == 'edit') {
            $data->edit = true;
        }
        else if($this->action == 'create') {
            $data->create = true;
        }
        else {
            $data->show = true;
        }
        return $data;
    }
}

class view_fill_page implements renderable, templatable {
    /** @var $cm Course module */
    var $cm = null;

    /** @var $otopo Otopo module instance */
    var $otopo = null;

    /** @var $action */
    var $action = 'progress';

    /** @var $session_id */
    var $session_id = null;

    /** @var $session_index */
    var $session_index = null;

    /** @var $session */
    var $session = null;

    /** @var $session_valid */
    var $session_valid = false;

    /** @var $session_valid_or_closed */
    var $session_valid_or_closed = false;

    public function __construct($cm, $otopo, $action, $session_id, $session_index = 0, $session_valid = false, $session_valid_or_closed = false, $session = null) {
        $this->action = $action;
        $this->cm = $cm;
        $this->otopo = $otopo;
        $this->session_id = $session_id;
        $this->session_index = $session_index;
        $this->session = $session;
        $this->session_valid = $session_valid;
        $this->session_valid_or_closed = $session_valid_or_closed;
    }

    public function export_for_template(renderer_base $output) {
        global $USER, $PAGE;

        $data = new stdClass();
        $data->progress = false;
        $data->evaluate = false;
        $data->evolution = false;
        switch($this->action) {
        case 'progress':
            $data->progress = true;
            break;
        case 'evolution':
            $data->evolution = true;
            break;
        case 'evaluate':
            $data->evaluate = true;
            break;
        }
        $data->action_ok = $data->progress || $data->evolution || $data->evaluate;
        $data->progress_or_evolution = $data->progress || $data->evolution;
        $data->cm = $this->cm;
        $data->otopo = $this->otopo;
        $data->otopos = get_user_otopos($this->otopo, $USER);
        $items = get_items_sorted_from_otopo($this->otopo->id);

        $data->nbr_degrees_max = table_items($items);
        $data->items = array_values($items);
        $data->image_mr_otopo = $output->image_url('mr_otopo', 'mod_otopo');
        $data->avatar = $output->user_picture($USER, array('size'=>150));
        $data->fullname = fullname($USER);
        $data->fromdate = userdate($this->otopo->allowsubmissionfromdate, get_string('strftimedatetimeshort', 'core_langconfig'));
        $data->todate = userdate($this->otopo->allowsubmissiontodate, get_string('strftimedatetimeshort', 'core_langconfig'));
        $data->session = $this->session;
        if($this->session) {
            $data->sessionfromdate = userdate($this->session->allowsubmissionfromdate, get_string('strftimedatetimeshort', 'core_langconfig'));
            $data->sessiontodate = userdate($this->session->allowsubmissiontodate, get_string('strftimedatetimeshort', 'core_langconfig'));
        }
        $data->is_open = is_open($this->otopo);
        $data->session_index = abs($this->session_index);
        $data->session_id = $this->session_id;
        $data->session_valid = $this->session_valid;
        $data->session_valid_or_closed = $this->session_valid_or_closed;
        $data->haslastmodificationonsession = last_modification_on_session($this->otopo, $USER, $this->session_id);
        $data->lastmodificationonsession = userdate($data->haslastmodificationonsession, get_string('strftimedatetimeshort', 'core_langconfig'));
        $data->haslastmodification = last_modification($this->otopo, $USER, $this->session_id);
        $data->lastmodification = userdate($data->haslastmodification, get_string('strftimedatetimeshort', 'core_langconfig'));

        $sessions = prepare_data($this->otopo, $items, $data->otopos, $USER);
        $data->show_otopos = !empty($data->otopos) || session_is_valid_or_closed($this->otopo->id, $USER, $this->otopo->session ? array_values($sessions)[0]->id : -1);

        if($this->action == 'progress') {
            $current_session = get_current_session($this->otopo, $USER);
            if($current_session) {
                $data->current_session_available = true;
            }
            else {
                $data->current_session_available = false;
            }
            foreach($items as $item) {
                $item->itemColor = $item->color;
                $item->itemName = $item->name;
            }
            $data->sessions = array_values($sessions);
            foreach($items as $item) {
                $item->results = [];
                foreach($sessions as $session) {
                    if(array_key_exists($item->id, $data->otopos) && array_key_exists($session->id, $data->otopos[$item->id])) {
                        $otopo = &$data->otopos[$item->id][$session->id];
                        $otopo->width = ceil($otopo->rank == 0 ? 0 : $otopo->rank / count($item->degrees) * 100);
                        $item->results[] = $otopo;
                    }
                    else {
                        $item->results[] = null;
                    }
                }
            }
            $data->star = $output->image_url('star', 'mod_otopo')->out();
            $data->star_container = $output->image_url('star_container', 'mod_otopo')->out();
            $data->help = $output->image_url('help2', 'mod_otopo')->out();
        }

        if($this->action == 'evolution') {
            $data->visual_radar = $this->otopo->sessionvisual == 0 ? true :  false;
            $data->visual_bar = $this->otopo->sessionvisual == 1 ? true :  false;
            $data->star = $output->image_url('star', 'mod_otopo')->out();
        }

        return $data;
    }
}

class mod_otopo_renderer extends plugin_renderer_base {
    /**
     * Defer to template.
     *
     * @param view_page $page
     *
     * @return string html for the page
     */
    public function render_view_page($page) {
        $data = $page->export_for_template($this);
        return parent::render_from_template('mod_otopo/view_page', $data);
    }

    /**
     * Defer to template.
     *
     * @param templates_page $page
     *
     * @return string html for the page
     */
    public function render_templates_page($page) {
        $data = $page->export_for_template($this);
        return parent::render_from_template('mod_otopo/templates_page', $data);
    }

    /**
     * Defer to template.
     *
     * @param view_fill_page $page
     *
     * @return string html for the page
     */
    public function render_view_fill_page($page) {
        $data = $page->export_for_template($this);
        return parent::render_from_template('mod_otopo/view_fill_page', $data);
    }

    /**
     * Defer to template..
     *
     * @param grading_app $app - All the data to render the grading app.
     */
    public function render_grading_app(grading_app $app) {
        $context = $app->export_for_template($this);
        return $this->render_from_template('mod_otopo/grading_app', $context);
    }
}
?>
