<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * External functions for webservice
 *
 * @package     mod_otopo
 * @copyright   2022 Kosmos <moodle@kosmos.fr>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
global $CFG;

require_once("$CFG->libdir/externallib.php");
require_once("$CFG->dirroot/user/externallib.php");
require_once(__DIR__.'/locallib.php');
require_once(__DIR__.'/lib.php');
require_once(__DIR__.'/grade_form.php');

class mod_otopo_external extends external_api {

    protected static $modulecontext = 0;

    public static function validate_otopo($otopo_id, $admin, $cmid = 0) {
        global $DB;

        if($otopo_id >= 0) {
            $otopo = $DB->get_record('otopo', array('id' => $otopo_id), '*', MUST_EXIST);
            $context = context_course::instance($otopo->course);
            $cm = get_coursemodule_from_instance('otopo', $otopo->id, $otopo->course, false, MUST_EXIST);
            self::$modulecontext = context_module::instance($cm->id);
            self::validate_context($context);
            if($admin) {
                require_capability('mod/otopo:admin', $context);
            }
            else {
                require_capability('mod/otopo:view', $context);
            }
        }
        else {
            $systemcontext = context_system::instance();   
            if($cmid) {
                $context = context_module::instance($cmid);
            }
            else {
                $context = $systemcontext;
            }
            self::validate_context($systemcontext);
            require_capability('mod/otopo:managetemplates', $context);
        }

        return $context;
    }

    public static function validate_user_otopo($otopo_id, $session_id, $write) {
        global $DB, $USER;

        if($otopo_id <= 0)
            throw new invalid_parameter_exception('Cannot evaluate template');

        $otopo = $DB->get_record('otopo', array('id' => $otopo_id), '*', MUST_EXIST);

        if ($session_id > 0) {
            $session = $DB->get_record('otopo_session', array('id' => $session_id), '*', MUST_EXIST);
            if ($session->otopo != $otopo_id)
                throw new invalid_parameter_exception('Cannot evaluate on session in another activity.');
        }

        if($write) {
            if(!is_open($otopo)) {
                throw new invalid_parameter_exception('Cannot evaluate on closed otopo.');
            }
            if(session_is_valid_or_closed($otopo_id, $USER, $session_id)) {
                throw new invalid_parameter_exception('Cannot evaluate on already envaluated session.');
            }
        }

        $context = context_course::instance($otopo->course);
        $cm = get_coursemodule_from_instance('otopo', $otopo->id, $otopo->course, false, MUST_EXIST);
        self::$modulecontext = context_module::instance($cm->id);
        self::validate_context($context);
        require_capability('mod/otopo:fill', $context);
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function get_items_parameters() {
        return new external_function_parameters(
            array(
                'otopo' => new external_value(PARAM_INT, 'id of the otopo activity'),
                'cmid' => new external_value(PARAM_INT, 'cmid of the otopo activity user come from', VALUE_OPTIONAL)
            )
        );
    }

    public static function get_items_returns() {
        return new external_multiple_structure(
            new external_single_structure(
                array(
                    'id' => new external_value(PARAM_INT, 'item record id'),
                    'name' => new external_value(PARAM_TEXT, 'item name'),
                    'color' => new external_value(PARAM_TEXT, 'item color'),
                    'ord' => new external_value(PARAM_INT, 'item record ord'),
                    'degrees' => new external_multiple_structure(
                        new external_single_structure(
                            array(
                                'id' => new external_value(PARAM_INT, 'degree record id'),
                                'name' => new external_value(PARAM_TEXT, 'degree name'),
                                'description' => new external_value(PARAM_TEXT, 'degree description', VALUE_OPTIONAL),
                                'grade' => new external_value(PARAM_INT, 'degree grade'),
                                'ord' => new external_value(PARAM_INT, 'degree record ord'),
                            )
                        )
                    )
                )
            )
        );
    }

    /**
     * Get items
     * @param int $otopo Otopo activity id
     * @return Items of the activity
     */
    public static function get_items($otopo, $cmid = 0) { //Don't forget to set it as static
        global $CFG, $DB;
        require_once("$CFG->dirroot/group/lib.php");

        $params = self::validate_parameters(self::get_items_parameters(), array('otopo'=>$otopo, 'cmid' => $cmid));

        $otopo_id = $params['otopo'];

        self::validate_otopo($otopo_id, false, $params['cmid']);

        return get_items_sorted_from_otopo($otopo_id);
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function create_item_parameters() {
        return new external_function_parameters(
            array(
                'otopo' => new external_value(PARAM_INT, 'id of the otopo activity'),
                'item' => new external_single_structure(
                    array(
                        'name' => new external_value(PARAM_TEXT, 'item name'),
                        'color' => new external_value(PARAM_TEXT, 'item color'),
                        'ord' => new external_value(PARAM_INT, 'degree record ord')
                    )
                ),
                'cmid' => new external_value(PARAM_INT, 'cmid of the otopo activity user come from', VALUE_OPTIONAL)
            )
        );
    }

    public static function create_item_returns() {
        return new external_value(PARAM_INT, 'item record id');
    }

  /**
     * Create item
     * @param int $otopo Otopo activity id
     * @param object $item Otopo item
     * @return Item created
     */
    public static function create_item($otopo, $item, $cmid) {
        global $CFG, $DB;
        require_once("$CFG->dirroot/group/lib.php");

        $params = self::validate_parameters(self::create_item_parameters(), array('otopo'=>$otopo, 'item' => $item, 'cmid' => $cmid));

        $otopo_id = $params['otopo'];

        self::validate_otopo($otopo_id, true, $params['cmid']);

        if(!has_otopo($otopo_id)) {
            $params['item']['otopo'] = $params['otopo'];

            $event = \mod_otopo\event\activity_updated::create(array('context' => self::$modulecontext));
            $event->trigger();

            return $DB->insert_record('otopo_item', $params['item']);
        }
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function edit_item_parameters() {
        return new external_function_parameters(
            array(
                'item' => new external_single_structure(
                    array(
                        'id' => new external_value(PARAM_INT, 'item id'),
                        'name' => new external_value(PARAM_TEXT, 'item name'),
                        'color' => new external_value(PARAM_TEXT, 'item color'),
                        'ord' => new external_value(PARAM_INT, 'degree record ord')
                    )
                ),
                'cmid' => new external_value(PARAM_INT, 'cmid of the otopo activity user come from', VALUE_OPTIONAL),
            )
        );
    }

    public static function edit_item_returns() {
        return null;
    }

    /**
     * Edit item
     * @param object $item Otopo item
     */
    public static function edit_item($item, $cmid) { //Don't forget to set it as static
        global $CFG, $DB;
        require_once("$CFG->dirroot/group/lib.php");

        $params = self::validate_parameters(self::edit_item_parameters(), array('item' => $item, 'cmid' => $cmid));

        $it = $DB->get_record('otopo_item', array('id' => $params['item']['id']));
        $otopo_id = $it->otopo;

        self::validate_otopo($otopo_id, true, $params['cmid']);

        if(has_otopo($otopo_id) && $it->ord != $params['item']['ord']) {
            return;
        }

        $event = \mod_otopo\event\activity_updated::create(array('context' => self::$modulecontext));
        $event->trigger();

        $DB->update_record('otopo_item', $params['item']);
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function delete_item_parameters() {
        return new external_function_parameters(
            array(
                'item_id' => new external_value(PARAM_INT, 'item id'),
                'cmid' => new external_value(PARAM_INT, 'cmid of the otopo activity user come from', VALUE_OPTIONAL)
            )
        );
    }

    public static function delete_item_returns() {
        return null;
    }

    /**
     * Delete item
     * @param int $item Otopo item id
     */
    public static function delete_item($item_id, $cmid) { //Don't forget to set it as static
        global $CFG, $DB;
        require_once("$CFG->dirroot/group/lib.php");

        $params = self::validate_parameters(self::delete_item_parameters(), array('item_id' => $item_id, 'cmid' => $cmid));

        $otopo_id = $DB->get_record('otopo_item', array('id' => $params['item_id']))->otopo;

        self::validate_otopo($otopo_id, true, $params['cmid']);

        if(!has_otopo($otopo_id)) {
            $event = \mod_otopo\event\activity_updated::create(array('context' => self::$modulecontext));
            $event->trigger();
            $DB->delete_records('otopo_item_degree', array('item' => $params['item_id']));
            $DB->delete_records('otopo_item', array('id' => $params['item_id']));
        }
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function create_degree_parameters() {
        return new external_function_parameters(
            array(
                'item_id' => new external_value(PARAM_INT, 'id of the otopo item'),
                'degree' => new external_single_structure(
                    array(
                        'name' => new external_value(PARAM_TEXT, 'degree name'),
                        'description' => new external_value(PARAM_TEXT, 'degree description', VALUE_OPTIONAL),
                        'grade' => new external_value(PARAM_INT, 'degree grade'),
                        'ord' => new external_value(PARAM_INT, 'degree record ord')
                    )
                ),
                'cmid' => new external_value(PARAM_INT, 'cmid of the otopo activity user come from', VALUE_OPTIONAL)
            )
        );
    }

    public static function create_degree_returns() {
        return new external_value(PARAM_INT, 'degree record id');
    }

    /**
     * Create degree
     * @param int $otopo Otopo activity id
     * @param object $degree Otopo degree
     * @return Degree created
     */
    public static function create_degree($item_id, $degree, $cmid) { //Don't forget to set it as static
        global $CFG, $DB;
        require_once("$CFG->dirroot/group/lib.php");

        $params = self::validate_parameters(self::create_degree_parameters(), array('item_id'=>$item_id, 'degree' => $degree, 'cmid' => $cmid));

        $item = $DB->get_record('otopo_item', array('id' => $params['item_id']), '*', MUST_EXIST);

        $otopo_id = $item->otopo;

        self::validate_otopo($otopo_id, true, $params['cmid']);

        if(!has_otopo($otopo_id)) {
            $params['degree']['item'] = $params['item_id'];

            $event = \mod_otopo\event\activity_updated::create(array('context' => self::$modulecontext));
            $event->trigger();

            return $DB->insert_record('otopo_item_degree', $params['degree']);
        }
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function edit_degree_parameters() {
        return new external_function_parameters(
            array(
                'degree' => new external_single_structure(
                    array(
                        'id' => new external_value(PARAM_INT, 'degree id'),
                        'name' => new external_value(PARAM_TEXT, 'degree name'),
                        'description' => new external_value(PARAM_TEXT, 'degree description', VALUE_OPTIONAL),
                        'grade' => new external_value(PARAM_INT, 'degree grade'),
                        'ord' => new external_value(PARAM_INT, 'degree record ord')
                    )
                ),
                'cmid' => new external_value(PARAM_INT, 'cmid of the otopo activity user come from', VALUE_OPTIONAL)
            )
        );
    }

    public static function edit_degree_returns() {
        return null;
    }

    /**
     * Edit degree
     * @param object $degree Otopo degree
     */
    public static function edit_degree($degree, $cmid) { //Don't forget to set it as static
        global $CFG, $DB;
        require_once("$CFG->dirroot/group/lib.php");

        $params = self::validate_parameters(self::edit_degree_parameters(), array('degree' => $degree, 'cmid' => $cmid));

        $deg = $DB->get_record('otopo_item_degree', array('id' => $params['degree']['id']));
        $item_id = $deg->item;
        $otopo_id = $DB->get_record('otopo_item', array('id' => $item_id))->otopo;

        self::validate_otopo($otopo_id, true, $params['cmid']);

        if(has_otopo($otopo_id) && ($deg->ord != $params['degree']['ord'] || $deg->grade != $params['degree']['grade'])) {
            return;
        }

        $event = \mod_otopo\event\activity_updated::create(array('context' => self::$modulecontext));
        $event->trigger();

        $DB->update_record('otopo_item_degree', $params['degree']);
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function delete_degree_parameters() {
        return new external_function_parameters(
            array(
                'degree_id' => new external_value(PARAM_INT, 'degree id'),
                'cmid' => new external_value(PARAM_INT, 'cmid of the otopo activity user come from', VALUE_OPTIONAL)
            )
        );
    }

    public static function delete_degree_returns() {
        return null;
    }

    /**
     * Delete degree
     * @param int $degree Otopo degree id
     */
    public static function delete_degree($degree_id, $cmid) { //Don't forget to set it as static
        global $CFG, $DB;
        require_once("$CFG->dirroot/group/lib.php");

        $params = self::validate_parameters(self::delete_degree_parameters(), array('degree_id' => $degree_id, 'cmid' => $cmid));

        $item_id = $DB->get_record('otopo_item_degree', array('id' => $params['degree_id']))->item;
        $otopo_id = $DB->get_record('otopo_item', array('id' => $item_id))->otopo;

        self::validate_otopo($otopo_id, true, $params['cmid']);

        if(!has_otopo($otopo_id)) {
            $event = \mod_otopo\event\activity_updated::create(array('context' => self::$modulecontext));
            $event->trigger();

            $DB->delete_records('otopo_item_degree', array('id' => $params['degree_id']));
        }
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function get_user_otopo_parameters() {
        return new external_function_parameters(
            array(
                'otopo' => new external_value(PARAM_INT, 'id of the otopo activity'),
                'session' => new external_value(PARAM_INT, 'id of the otopo activity session'),
            )
        );
    }

    public static function get_user_otopo_returns() {
        return new external_multiple_structure(
            new external_single_structure(
                array(
                    'id' => new external_value(PARAM_INT, 'otopo record id', VALUE_OPTIONAL),
                    'item' => new external_value(PARAM_INT, 'otopo item'),
                    'degree' => new external_value(PARAM_INT, 'otopo degree'),
                    'justification' => new external_value(PARAM_TEXT, 'otopo justification'),
                    'comment' => new external_value(PARAM_TEXT, 'otopo teacher comment', VALUE_OPTIONAL)
                )
            )
        );
    }

    /**
     * Get user otopo
     * @param int $otopo Otopo activity id
     * @param int $session Otopo activity session id
     * @return Otopos of the user session
     */
    public static function get_user_otopo($otopo, $session) { //Don't forget to set it as static
        global $CFG, $DB, $USER;
        require_once("$CFG->dirroot/group/lib.php");

        $params = self::validate_parameters(self::get_user_otopo_parameters(), array('otopo'=>$otopo, 'session'=>$session));

        $otopo_id = $params['otopo'];
        $o = $DB->get_record('otopo', array('id' => $otopo_id), '*', MUST_EXIST);
        $session_id = $params['session'];

        self::validate_user_otopo($otopo_id, $session_id, false);

        $result = $DB->get_records_sql('SELECT {otopo_user_otopo}.id AS id,item,degree,justification,teacher_comment AS comment FROM {otopo_user_otopo} INNER JOIN {otopo_item} AS it ON item = it.id WHERE userid = :user AND session = :session AND it.otopo = :otopo', array('user' => intval($USER->id), 'session' => $session_id, 'otopo' => $otopo_id));

        # Si aucun positionnement
        # on retourne les resultats de la dernière session validée
        $initial_session_id = $session_id;
        $session_id = null;
        if(empty($result)) {
            if($o->session) {
                $session_id = get_last_session_closed($o)->id;
            }
            else if($initial_session_id < -1) {
                $otopos = get_users_otopos($o, [$USER->id]);
                $session_id = $initial_session_id + 1;
                //foreach($otopos[$USER->id] as $session => $otopo) {
                //    if(session_is_valid($o->id, $USER, $session)) {
                //        if($session_id == null || $session < $session_id) {
                //            $session_id = $session;
                //        }
                //    }
                //}
            }
        }
        if($session_id) {
            $result = $DB->get_records_sql('SELECT {otopo_user_otopo}.id AS id,item,degree,justification,teacher_comment AS comment FROM {otopo_user_otopo} INNER JOIN {otopo_item} AS it ON item = it.id WHERE userid = :user AND session = :session AND it.otopo = :otopo', array('user' => intval($USER->id), 'session' => $session_id, 'otopo' => $otopo_id));
            foreach($result as &$r) {
                $r->id = null;
            }
        }

        $o = $DB->get_record('otopo', array('id' => $otopo_id), '*', MUST_EXIST);
        if(!$o->showteachercomments) {
            foreach($result as &$r) {
                unset($r->comment);
            }
        }

        return $result;
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function set_user_otopo_parameters() {
        return new external_function_parameters(
            array(
                'otopo' => new external_value(PARAM_INT, 'id of the otopo activity'),
                'session' => new external_value(PARAM_INT, 'id of the otopo activity session'),
                'item' => new external_value(PARAM_INT, 'id of the otopo activity item'),
                'degree' => new external_value(PARAM_INT, 'id of the otopo activity item degree'),
                'justification' => new external_value(PARAM_TEXT, 'justification of the otopo')
            )
        );
    }

    public static function set_user_otopo_returns() {
        return null;
    }

    /**
     * Set user otopo
     * @param int $otopo Otopo activity id
     * @param int $session Otopo activity session id
     * @param int $item Otopo activity item
     * @param int $degree Otopo activity item degree
     * @param string $justification Otopo justification
     */
    public static function set_user_otopo($otopo, $session, $item, $degree, $justification) { //Don't forget to set it as static
        global $CFG, $DB, $USER;
        require_once("$CFG->dirroot/group/lib.php");

        $params = self::validate_parameters(self::set_user_otopo_parameters(), array('otopo'=>$otopo, 'session'=>$session, 'item'=>$item, 'degree'=>$degree, 'justification'=>$justification));

        $otopo_id = $params['otopo'];
        $session_id = $params['session'];
        $item_id = $params['item'];
        $degree_id = $params['degree'];
        $justification = $params['justification'];
        $date = new DateTime();
        $lastmodificationdate = $date->getTimestamp();

        self::validate_user_otopo($otopo_id, $session_id, true);

        $user_otopo = $DB->get_record('otopo_user_otopo', array('userid' => $USER->id, 'session' => $session_id, 'item' => $item_id));
        if($user_otopo) {
            if($degree_id) {
                $user_otopo->degree = $degree_id;
                $user_otopo->justification = $justification;
                $user_otopo->lastmodificationdate = $lastmodificationdate;
                $DB->update_record('otopo_user_otopo', $user_otopo);
            }
            else {
                $DB->delete_records('otopo_user_otopo', array('id' => $user_otopo->id));
            }
        }
        else {
            $user_otopo = (object)array('userid' => $USER->id, 'session' => $session_id, 'item' => $item_id, 'degree' => $degree_id, 'justification' => $justification, 'lastmodificationdate' => $lastmodificationdate);
            $DB->insert_record('otopo_user_otopo', $user_otopo);
        }

        $event = \mod_otopo\event\session_saved::create(array('context' => self::$modulecontext));
        $event->trigger();


        return;
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function get_group_chart_parameters() {
        return new external_function_parameters(
            array(
                'otopo' => new external_value(PARAM_INT, 'id of the otopo activity'),
                'users' => new external_multiple_structure(
                    new external_value(PARAM_INT, 'id of user')
                ),
                'session' => new external_value(PARAM_INT, 'id of the session', VALUE_OPTIONAL),
            )
        );
    }

    public static function get_group_chart_returns() {
        return new external_single_structure(
            array(
                'labels' => new external_multiple_structure(
                    new external_value(PARAM_TEXT, 'label')
                ),
                'fullLabels' => new external_multiple_structure(
                    new external_value(PARAM_TEXT, 'label')
                ),
                'datasets' => new external_multiple_structure(
                    new external_single_structure(
                        array(
                            'label' => new external_value(PARAM_TEXT, 'label'),
                            'borderWidth' => new external_value(PARAM_TEXT, 'border width'),
                            'data' => new external_multiple_structure(
                                new external_value(PARAM_TEXT, 'numeric value')
                            ),
                            'dataSource' => new external_multiple_structure(
                                new external_value(PARAM_INT, 'numeric value')
                            ),
                            'backgroundColor' => new external_multiple_structure(
                                new external_value(PARAM_TEXT, 'color hex')
                            ),
                            'borderColor' => new external_multiple_structure(
                                new external_value(PARAM_TEXT, 'color hex')
                            ),
                            'hoverBackgroundColor' => new external_multiple_structure(
                                new external_value(PARAM_TEXT, 'color hex')
                            )
                        )
                    )
                )
            )
        );
    }

    /**
     * Get group chart
     * @param int $otopo Otopo activity id
     * @param array $users users to show
     */
    public static function get_group_chart($o, $users, $session) {
        global $DB;

        $params = self::validate_parameters(self::get_group_chart_parameters(), array('otopo'=>$o, 'users' => $users, 'session' => $session));

        $otopo_id = $params['otopo'];
        $o = $DB->get_record('otopo', array('id' => $otopo_id), '*', MUST_EXIST);
        $users = $params['users'];
        $session = $params['session'];

        self::validate_otopo($otopo_id, true);

        $distribution = get_distribution_by_item($o, $users, $session);
        $items = get_items_sorted_from_otopo($o->id);

        $chart = array(
            'labels' => [],
            'datasets' => []
        );

        $nbr_degrees_max = 0;
        foreach($items as $item) {
            if(count($item->degrees) > $nbr_degrees_max) {
                $nbr_degrees_max = count($item->degrees);
            }
        }
        foreach(array_values($items) as $key1 => $item) {
            $chart['labels'][] = get_string('item_item', 'otopo') . ' ' . ($key1 + 1);
            $chart['fullLabels'][] = $item->name;
            for($degree = 0; $degree < $nbr_degrees_max; $degree++) {
                if(!array_key_exists($degree, $chart['datasets'])) {
                    $chart['datasets'][$degree]['label'] = get_string('autoeval_degree', 'otopo') . ' ' . ($degree + 1);
                    $chart['datasets'][$degree]['data'] = [];
                    $chart['datasets'][$degree]['dataSource'] = [];
                    $chart['datasets'][$degree]['backgroundColor'] = [];
                    $chart['datasets'][$degree]['hoverBackgroundColor'] = [];
                    $chart['datasets'][$degree]['borderColor'] = [];
                    $chart['datasets'][$degree]['borderWidth'] = 1;
                }
                if(!array_key_exists($key1, $chart['datasets'][$degree]['data'])) {
                    $chart['datasets'][$degree]['dataSource'][$key1] = 0;
                    $chart['datasets'][$degree]['data'][$key1] = 0;
                    $chart['datasets'][$degree]['backgroundColor'][$key1] = "#323e70" . dechex((1 - (($degree + 1) / ($nbr_degrees_max + 2))) * 255);
                    $chart['datasets'][$degree]['hoverBackgroundColor'][$key1] = "#323e70ff";
                    $chart['datasets'][$degree]['borderColor'][$key1] = "#323e70ff";
                    $chart['datasets'][$degree]['toto'] = $item->name;
                }
                if(array_key_exists($degree, $distribution) && array_key_exists($key1, $distribution[$degree])) {
                    $chart['datasets'][$degree]['dataSource'][$key1] = $distribution[$degree][$key1];
                    $chart['datasets'][$degree]['data'][$key1] = ceil($chart['datasets'][$degree]['dataSource'][$key1] * 100 / count($users));
                    $chart['datasets'][$degree]['toto'] = $item->name;
                }
            }
        }

        return $chart;
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function get_my_evolution_parameters() {
        return new external_function_parameters(
            array(
                'otopo' => new external_value(PARAM_INT, 'id of the otopo activity'),
                'visual' => new external_value(PARAM_TEXT, 'visual for charts'),
                'item' => new external_value(PARAM_INT, 'id of the item to focus evolution', VALUE_OPTIONAL),
            )
        );
    }

    public static function get_chart_return() {
        return new external_single_structure(
            array(
                'id' => new external_value(PARAM_INT, 'id of session', VALUE_OPTIONAL),
                'grade' => new external_value(PARAM_INT, 'grade', VALUE_OPTIONAL),
                'comment' => new external_value(PARAM_RAW, 'comment', VALUE_OPTIONAL),
                'label' => new external_value(PARAM_TEXT, 'label', VALUE_OPTIONAL),
                'fullLabel' => new external_value(PARAM_TEXT, 'full label', VALUE_OPTIONAL),
                'labels' => new external_multiple_structure(
                    new external_value(PARAM_TEXT, 'label', VALUE_OPTIONAL)
                ),
                'fullLabels' => new external_multiple_structure(
                    new external_value(PARAM_TEXT, 'label', VALUE_OPTIONAL)
                ),
                'color' => new external_value(PARAM_TEXT, 'color hex', VALUE_OPTIONAL),
                'allowsubmissionfromdate' => new external_value(PARAM_TEXT, 'from date', VALUE_OPTIONAL),
                'allowsubmissiontodate' => new external_value(PARAM_TEXT, 'to date', VALUE_OPTIONAL),
                'datasets' => new external_multiple_structure(
                    new external_single_structure(
                        array(
                            'label' => new external_value(PARAM_TEXT, 'label', VALUE_OPTIONAL),
                            'borderWidth' => new external_value(PARAM_TEXT, 'border width', VALUE_OPTIONAL),
                            'data' => new external_multiple_structure(
                                new external_value(PARAM_INT, 'numeric value', VALUE_OPTIONAL)
                            ),
                            'labels' => new external_multiple_structure(
                                new external_value(PARAM_TEXT, 'label', VALUE_OPTIONAL)
                            ),
                            'backgroundColor' => new external_multiple_structure(
                                new external_value(PARAM_TEXT, 'color hex', VALUE_OPTIONAL)
                            ),
                            'hoverBackgroundColor' => new external_multiple_structure(
                                new external_value(PARAM_TEXT, 'color hex', VALUE_OPTIONAL)
                            ),
                            'borderColor' => new external_multiple_structure(
                                new external_value(PARAM_TEXT, 'color hex', VALUE_OPTIONAL)
                            )
                        )
                    )
                )
            )
        );
    }

    public static function get_my_evolution_returns() {
        return new external_single_structure(
            array(
                'current_chart' => self::get_chart_return(),
                'chart_item' => self::get_chart_return(),
                'charts' => new external_multiple_structure(
                    self::get_chart_return()
                ),
                'chart_item_degrees' => new external_multiple_structure(
                    new external_single_structure(
                        array(
                            'label' => new external_value(PARAM_TEXT, 'label'),
                            'degree' => new external_single_structure(
                                array(
                                    'id' => new external_value(PARAM_INT, 'id of degree'),
                                    'name' => new external_value(PARAM_TEXT, 'name'),
                                    'description' => new external_value(PARAM_TEXT, 'description'),
                                    'grade' => new external_value(PARAM_INT, 'grade'),
                                    'ord' => new external_value(PARAM_INT, 'ord')
                                )
                            ),
                            'color' => new external_value(PARAM_TEXT, 'color hex'),
                            'allowsubmissionfromdate' => new external_value(PARAM_TEXT, 'from date'),
                            'allowsubmissiontodate' => new external_value(PARAM_TEXT, 'to date'),
                            'lastmodificationonsession' => new external_value(PARAM_TEXT, 'last modification')
                        )
                    )
                ),
                'max' => new external_value(PARAM_INT, 'max degrees')
            )
        );
    }

    /**
     * Get group chart
     * @param int $otopo Otopo activity id
     * @param array $users users to show
     */
    public static function get_my_evolution($o, $visual, $item) {
        global $DB, $USER;

        $params = self::validate_parameters(self::get_my_evolution_parameters(), array('otopo'=>$o, 'visual' => $visual, 'item' => $item));

        $otopo_id = $params['otopo'];
        $o = $DB->get_record('otopo', array('id' => $otopo_id), '*', MUST_EXIST);
        $visual = $params['visual'];
        $item = $params['item'];

        self::validate_otopo($otopo_id, false);

        $items = get_items_from_otopo($o->id);
        $items_sorted = get_items_sorted_from_otopo($o->id);
        $otopos = get_user_otopos($o, $USER);
        $sessions = prepare_data($o, $items_sorted, $otopos, $USER);

        $data = get_my_evolution($o, $items, $items_sorted, $sessions, $otopos, $USER, $visual, $item);

        $current = get_current_session($o, $USER);

        if ($current && property_exists($data, 'charts') && array_key_exists(abs($current[0]) - 1, $data->charts)) {
            $current_chart = $data->charts[abs($current[0]) - 1];
            unset($data->charts[abs($current[0]) - 1]);
        } else {
            $current_chart = null;
        }

        if($o->gradeonlyforteacher) {
            if($current_chart) {
                $current_chart['grade'] = null;
            }
            foreach($data->charts as &$chart) {
                $chart['grade'] = null;
            }
        }

        return array(
            'current_chart' => $current_chart ? $current_chart : ['labels' => [], 'fullLabels' => [], 'datasets' => []],
            'charts' => $data->charts ? $data->charts : [],
            'chart_item' => property_exists($data, 'chart_item') ? $data->chart_item : ['labels' => [], 'fullLabels' => [], 'datasets' => []],
            'chart_item_degrees' => property_exists($data, 'chart_item_degrees') ? $data->chart_item_degrees : [],
            'max' => $data->max
        );
    }

    /**
     * Returns description of method parameters
     *
     * @return external_function_parameters
     * @since Moodle 3.1
     */
    public static function list_participants_parameters() {
        return new external_function_parameters(
            array(
                'otopo' => new external_value(PARAM_INT, 'otopo instance id'),
                'filter' => new external_value(PARAM_RAW, 'search string to filter the results'),
                'skip' => new external_value(PARAM_INT, 'number of records to skip', VALUE_DEFAULT, 0),
                'limit' => new external_value(PARAM_INT, 'maximum number of records to return', VALUE_DEFAULT, 0)
            )
        );
    }

    /**
     * Retrieves the list of students to be graded
     *
     * @param int $otopo the otopo instance id
     * @return array of warnings and status result
     * @since Moodle 3.1
     * @throws moodle_exception
     */
    public static function list_participants($o, $filter) {
        global $DB, $CFG;

        $params = self::validate_parameters(self::list_participants_parameters(), array(
            'otopo'=>$o,
            'filter' => $filter,
            'skip' => $skip,
            'limit' => $limit
        ));

        $otopo_id = $params['otopo'];
        $o = $DB->get_record('otopo', array('id' => $otopo_id), '*', MUST_EXIST);

        $context = self::validate_otopo($otopo_id, false);

        $participants = get_participants($o, self::$modulecontext);

        require_once($CFG->dirroot.'/user/lib.php');
        $userfields = user_get_default_fields();
        // Remove enrolled courses from users fields to be returned.
        $key = array_search('enrolledcourses', $userfields);
        if ($key !== false) {
            unset($userfields[$key]);
        } else {
            throw new moodle_exception('invaliduserfield', 'error', '', 'enrolledcourses');
        }

        $result = array();
        $index = 0;
        foreach ($participants as $record) {
            // Preserve the fullname set by the assignment.
            $fullname = fullname($record, has_capability('moodle/site:viewfullnames', $context));
            $searchable = $fullname;
            $match = false;
            if (empty($filter)) {
                $match = true;
            } else {
                $filter = core_text::strtolower($filter);
                $value = core_text::strtolower($searchable);
                if (is_string($value) && (core_text::strpos($value, $filter) !== false)) {
                    $match = true;
                }
            }
            if ($match) {
                $index++;
                if ($index <= $params['skip']) {
                    continue;
                }
                if (($params['limit'] > 0) && (($index - $params['skip']) > $params['limit'])) {
                    break;
                }
                $userdetails = user_get_user_details($record, $course, $userfields);
                $userdetails['id'] = $record->id;
                $userdetails['fullname'] = $fullname;

                $result[] = $userdetails;
            }
        }
        return $result;
    }

    /**
     * Returns the description of the results of the mod_assign_external::list_participants() method.
     *
     * @return external_description
     * @since Moodle 3.1
     */
    public static function list_participants_returns() {
        // Get user description.
        $userdesc = core_user_external::user_description();
        // List unneeded properties.
        $unneededproperties = [
            'auth', 'confirmed', 'lang', 'calendartype', 'theme', 'timezone', 'mailformat'
        ];
        // Remove unneeded properties for consistency with the previous version.
        foreach ($unneededproperties as $prop) {
            unset($userdesc->keys[$prop]);
        }

        // Override property attributes for consistency with the previous version.
        $userdesc->keys['fullname']->type = PARAM_NOTAGS;
        $userdesc->keys['profileimageurlsmall']->required = VALUE_OPTIONAL;
        $userdesc->keys['profileimageurl']->required = VALUE_OPTIONAL;
        $userdesc->keys['email']->desc = 'Email address';
        $userdesc->keys['idnumber']->desc = 'The idnumber of the user';

        // Define other keys.
        /*$otherkeys = [
            'sessions' => new external_multiple_structure(
                new external_single_structure(
                    array(
                        'id' => new external_value(PARAM_INT, 'session id'),
                        'validated' => new external_value(PARAM_BOOL, 'have they validated their otopo session')
                    )
                )
            )
            ];*/

        // Merge keys.
        return new external_multiple_structure($userdesc);
    }

    public static function get_participant_parameters() {
        return new external_function_parameters(
            array(
                'otopo' => new external_value(PARAM_INT, 'otopo instance id'),
                'userid' => new external_value(PARAM_INT, 'user id'),
                'session' => new external_value(PARAM_INT, 'session id', VALUE_OPTIONAL)
            )
        );
    }

    /**
     * Get the user participating in the given assignment. An error with code 'usernotincourse'
     * is thrown is the user isn't a participant of the given assignment.
     *
     * @param int $otopo the otopo instance id
     * @param int $userid the user id
     * @param int $session the session id
     * @since Moodle 3.1
     * @throws moodle_exception
     */
    public static function get_participant($o, $userid, $session) {
        global $DB, $CFG;
        require_once($CFG->dirroot . "/user/lib.php");

        $params = self::validate_parameters(self::get_participant_parameters(), array(
            'otopo' => $o,
            'userid' => $userid,
            'session' => $session
        ));

        $otopo_id = $params['otopo'];
        $o = $DB->get_record('otopo', array('id' => $otopo_id), '*', MUST_EXIST);

        $context = self::validate_otopo($otopo_id, false);

        require_capability('mod/otopo:grade', $context);

        $participant = get_participant($o, self::$modulecontext, $userid);

        if (!$participant) {
            // No participant found so we can return early.
            throw new moodle_exception('usernotincourse');
        }



        $items = get_items_from_otopo($o->id);
        $items_sorted = get_items_sorted_from_otopo($o->id);
        $otopos = get_user_otopos($o, $participant);
        $sessions = prepare_data($o, $items_sorted, $otopos, $participant);
        $participant_sessions = $participant->sessions;
        $participant->sessions = [];
        foreach($sessions as $s1) {
            $found = false;
            foreach($participant_sessions as $s2) {
                if($s1->id == intval($s2['id'])) {
                    $found = true;
                    $s2['id'] = intval($s2['id']);
                    $participant->sessions[] = $s2;
                }
            }
            if(!$found) {
                $is_valid = session_is_valid_or_closed($o->id, $participant, $s1->id);
                if($is_valid) {
                    $participant->sessions[] = [
                        'id' => $s1->id,
                        'validated' => $is_valid,
                        'not_found' => true
                    ];
                }
            }
        }

        $session_o = null;
        $key_without_not_found = 1;
        $key = 1;
        foreach($participant->sessions as &$s) {
            if(!array_key_exists('not_found', $s)) {
                $s['key_without_not_found'] = $key_without_not_found;
                ++$key_without_not_found;
            }
            $s['key'] = $key;
            ++$key;
            if($s['id'] == $session) {
                $s['selected'] = true;
                $session_o = $s;
            }
            else {
                $s['selected'] = false;
            }
            unset($s['grade']);
            unset($s['comment']);
        }

        $data_sessions = get_my_evolution($o, $items, $items_sorted, $sessions, $otopos, $participant, $o->sessionvisual == 0 ? 'radar' : 'bar', null);
        $lastmodification = last_modification_on_session($o, $participant, $session_o['id']);
        if($lastmodification) {
            $session_o['lastmodification'] = userdate(last_modification_on_session($o, $participant, $session_o['id']), get_string('strftimedatetimeshort', 'core_langconfig'));
        }
        $session_o['grade'] = $sessions[$session_o['key']-1]->grade;
        $session_o['comment'] = $sessions[$session_o['key']-1]->comment;

        $session_chart = $data_sessions->charts[$session_o['key_without_not_found']-1];

        $items_otopos = [];

        $url = '~(?:(https?)://([^\s<]+)|(www\.[^\s<]+?\.[^\s<]+))(?<![\.,:])~i'; 
        foreach($items_sorted as $item) {
            $items_otopo = [
                'id' => $item->id,
                'key' => $item->ord,
                'name' => $item->name,
                'color' => $item->color,
            ];
            if(array_key_exists($item->id, $otopos) && array_key_exists($session, $otopos[$item->id])) {
                $otopo = $otopos[$item->id][$session];
                $items_otopo['otopo_id'] = $otopo->id;
                $items_otopo['otopo_teacher_comment'] = preg_replace($url, '<a href="$0" target="_blank" title="$0">$0</a>', $otopo->teacher_comment);
                $items_otopo['otopo_justification'] = preg_replace($url, '<a href="$0" target="_blank" title="$0">$0</a>', $otopo->justification);
                $items_otopo['otopo_degree_id'] = $otopo->degree->id;
                $items_otopo['otopo_degree_key'] = $otopo->rank;
                $items_otopo['otopo_degree_name'] = $otopo->degree->name;
                $items_otopo['otopo_degree_name'] = $otopo->degree->name;
                $items_otopo['otopo_degree_description'] = $otopo->degree->description;
                $items_otopo['otopo_degree_width'] = $otopo->rank == 0 ? 0 : $otopo->rank / count($item->degrees) * 100;
            }
            $items_otopos[] = $items_otopo;
        }

        $return = array(
            'id' => $participant->id,
            'fullname' => $participant->fullname,
            'sessions' => $participant->sessions ? $participant->sessions : [],
            'session' => $session_o,
            'session_chart' => $session_chart ? $session_chart : ['labels' => [], 'fullLabels' => [], 'datasets' => []],
            'items' => $items_otopos,
            'max' => $data_sessions->max
        );


        if ($userdetails = user_get_user_details($participant, $course)) {
            $return['user'] = $userdetails;
        }

        // Needed because of a breaking change in chartJS 3 which is introduce in moodle 4
        $moodleVersion = get_config('')->version;
        if ($moodleVersion < 2022041906) {
            $moodlePre4 = true;
        } else {
            $moodlePre4 = false;
        }
        $return['moodlePre4'] = $moodlePre4;

        return $return;
    }

    /**
     * Returns description of method result value
     *
     * @return external_description
     * @since Moodle 3.1
     */
    public static function get_participant_returns() {
        $userdescription = core_user_external::user_description();
        $userdescription->default = [];
        $userdescription->required = VALUE_OPTIONAL;

        return new external_single_structure(array(
            'id' => new external_value(PARAM_INT, 'ID of the user'),
            'fullname' => new external_value(PARAM_NOTAGS, 'The fullname of the user'),
            'user' => $userdescription,
            'session_chart' => self::get_chart_return(),
            'session' => new external_single_structure(
                array(
                    'id' => new external_value(PARAM_INT, 'session id', VALUE_OPTIONAL),
                    'key' => new external_value(PARAM_INT, 'session key', VALUE_OPTIONAL),
                    'validated' => new external_value(PARAM_BOOL, 'have they validated their otopo session', VALUE_OPTIONAL),
                    'grade' => new external_value(PARAM_INT, 'grade', VALUE_OPTIONAL),
                    'comment' => new external_value(PARAM_RAW, 'comment', VALUE_OPTIONAL),
                    'selected' => new external_value(PARAM_BOOL, 'selected', VALUE_OPTIONAL),
                    'lastmodification' => new external_value(PARAM_TEXT, 'last modification', VALUE_OPTIONAL),
                )
            ),
            'sessions' => new external_multiple_structure(
                new external_single_structure(
                    array(
                        'id' => new external_value(PARAM_INT, 'session id'),
                        'key' => new external_value(PARAM_INT, 'session key'),
                        'selected' => new external_value(PARAM_BOOL, 'selected'),
                        'validated' => new external_value(PARAM_BOOL, 'have they validated their otopo session', VALUE_OPTIONAL),
                    )
                )
            ),
            'items' => new external_multiple_structure(
                new external_single_structure(
                    array(
                        'id' => new external_value(PARAM_INT, 'item id'),
                        'key' => new external_value(PARAM_INT, 'item key'),
                        'name' => new external_value(PARAM_TEXT, 'item name'),
                        'color' => new external_value(PARAM_TEXT, 'item color'),
                        'otopo_id' => new external_value(PARAM_INT, 'otopo id', VALUE_OPTIONAL),
                        'otopo_justification' => new external_value(PARAM_RAW, 'otopo justification', VALUE_OPTIONAL),
                        'otopo_teacher_comment' => new external_value(PARAM_RAW, 'otopo comment', VALUE_OPTIONAL),
                        'otopo_degree_id' => new external_value(PARAM_INT, 'degree key', VALUE_OPTIONAL),
                        'otopo_degree_key' => new external_value(PARAM_INT, 'degree key', VALUE_OPTIONAL),
                        'otopo_degree_name' => new external_value(PARAM_TEXT, 'degree name', VALUE_OPTIONAL),
                        'otopo_degree_description' => new external_value(PARAM_TEXT, 'degree description', VALUE_OPTIONAL),
                        'otopo_degree_width' => new external_value(PARAM_TEXT, 'degree width', VALUE_OPTIONAL)
                    )
                )
            ),
            'max' => new external_value(PARAM_INT, 'max degrees'),
        ));
    }

    /**
     * Describes the parameters for submit_grading_form webservice.
     * @return external_function_parameters
     * @since  Moodle 3.1
     */
    public static function submit_grading_form_parameters() {
        return new external_function_parameters(
            array(
                'otopo' => new external_value(PARAM_INT, 'otopo instance id'),
                'userid' => new external_value(PARAM_INT, 'user id'),
                'session' => new external_value(PARAM_INT, 'session id', VALUE_OPTIONAL),
                'jsonformdata' => new external_value(PARAM_RAW, 'The data from the grading form, encoded as a json array'),
                'itemscomments' => new external_multiple_structure(
                    new external_single_structure(
                        array(
                            'id' => new external_value(PARAM_INT, 'item id'),
                            'value' => new external_value(PARAM_RAW, 'teacher comment')
                        )
                    )
                )
            )
        );
    }

    /**
     * Submit the logged in users assignment for grading.
     *
     * @param int $otopo the otopo instance id
     * @param int $userid the user id
     * @param int $session the session id
     * @param string $jsonformdata The data from the form, encoded as a json array.
     * @param array $itemscomments
     * @return array of warnings to indicate any errors.
     * @since Moodle 3.1
     */
    public static function submit_grading_form($o, $userid, $session, $jsonformdata, $itemscomments) {
        global $CFG, $USER, $DB;

        $params = self::validate_parameters(self::submit_grading_form_parameters(), array(
            'otopo' => $o,
            'userid' => $userid,
            'session' => $session,
            'jsonformdata' => $jsonformdata,
            'itemscomments' => $itemscomments
        ));

        $otopo_id = $params['otopo'];
        $o = $DB->get_record('otopo', array('id' => $otopo_id), '*', MUST_EXIST);

        $context = self::validate_otopo($otopo_id, false);

        require_capability('mod/otopo:grade', $context);

        $serialiseddata = json_decode($params['jsonformdata']);

        $data = array();
        parse_str($serialiseddata, $data);

        $warnings = array();

        // On teste si la session est cloturée
        if(!session_is_valid_or_closed($otopo_id, (object)array('id' => $userid), $session)) {
            $warnings[] = array(
                'item' => 'Invalid session.',
                'itemid' => $otopo_id,
                'warningcode' => 'couldnotsavegrade',
                'message' => 'Could not save grade'
            );
            return $warnings;
        }

        $customdata = (object)$data;
        $grader = $DB->get_record('otopo_grader', array('userid' => $userid, 'session' => $session, 'otopo' => $o->id));
        $formparams = array('otopo' => $o, 'grader' => $grader);

        // Data is injected into the form by the last param for the constructor.
        $mform = new grade_form(null, $formparams, 'post', '', null, true, $data);
        $validateddata = $mform->get_data();

        if ($validateddata && $validateddata->grade) {
            $grader = $DB->get_record('otopo_grader', array('userid' => $userid, 'session' => $session, 'otopo' => $o->id));
            if($grader) {
                $grader->comment = $validateddata->comment['text'];
                $grader->grade = $validateddata->grade;
                $DB->update_record('otopo_grader', $grader);
            }
            else {
                $DB->insert_record('otopo_grader', array('userid' => $userid, 'session' => $session, 'otopo' => $o->id, 'comment' => $validateddata->comment->text, 'grade' => $validateddata->grade));
            }
            otopo_update_grades($o, $userid);
        } else {
            $warnings[] = array(
                'item' => 'Form validation failed.',
                'itemid' => $otopo_id,
                'warningcode' => 'couldnotsavegrade',
                'message' => 'Could not save grade'
            );
        }

        foreach($itemscomments as $item_comment) {
            $otopo = $DB->get_record('otopo_user_otopo', array('userid' => $userid, 'session' => $session, 'item' => $item_comment['id']));
            $otopo->teacher_comment = $item_comment['value'];
            $DB->update_record('otopo_user_otopo', $otopo);
        }


        return $warnings;
    }

    /**
     * Describes the return for submit_grading_form
     * @return external_function_parameters
     * @since  Moodle 3.1
     */
    public static function submit_grading_form_returns() {
        return new external_warnings();
    }
}
?>
